<?php

namespace App\Http\Controllers\Admin\GeneratedHistory;

use App\Http\Controllers\Controller;
use App\Models\Sender;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Yajra\DataTables\Facades\DataTables;

class SenderController extends Controller
{
    public function index()
    {
        $data['senderRecord'] = Sender::selectRaw('COUNT(id) AS totalSender')
            ->selectRaw('COUNT(CASE WHEN is_verified = 1 THEN id END) AS verifiedSender')
            ->selectRaw('(COUNT(CASE WHEN is_verified = 1 THEN id END) / COUNT(id)) * 100 AS verifiedSenderPercentage')
            ->selectRaw('COUNT(CASE WHEN is_verified = 0 THEN id END) AS unverifiedSender')
            ->selectRaw('(COUNT(CASE WHEN is_verified = 0 THEN id END) / COUNT(id)) * 100 AS unverifiedSenderPercentage')
            ->selectRaw('COUNT(CASE WHEN DATE(created_at) = CURRENT_DATE THEN id END) AS todaySender')
            ->selectRaw('(COUNT(CASE WHEN DATE(created_at) = CURRENT_DATE THEN id END) / COUNT(id)) * 100 AS todaySenderPercentage')
            ->selectRaw('COUNT(CASE WHEN YEARWEEK(created_at, 1) = YEARWEEK(CURDATE(), 1) THEN id END) AS thisWeekSender')
            ->selectRaw('(COUNT(CASE WHEN YEARWEEK(created_at, 1) = YEARWEEK(CURDATE(), 1) THEN id END) / COUNT(id)) * 100 AS thisWeekSenderPercentage')
            ->get()
            ->toArray();

        return view('admin.sender.list', $data);
    }

    public function senderSearch(Request $request)
    {
        $filterName = $request->name;
        $filterCreator = $request->creator;
        $filterStatus = $request->filterStatus;
        $filterDate = explode('-', $request->filterDate);
        $startDate = $filterDate[0];
        $endDate = isset($filterDate[1]) ? trim($filterDate[1]) : null;

        $senders = Sender::with(['user:id,firstname,lastname,username,image,image_driver'])
            ->whereHas('user')
            ->orderBy('id', 'DESC')
            ->when(isset($filterName), function ($query) use ($filterName) {
                return $query->where('from_name', 'LIKE', '%' . $filterName . '%');
            })
            ->when(isset($filterCreator), function ($query) use ($filterCreator) {
                $query->whereHas('user', function ($qq) use ($filterCreator) {
                    $qq->where('firstname', 'LIKE', '%' . $filterCreator . '%')
                        ->orWhere('lastname', 'LIKE', '%' . $filterCreator . '%')
                        ->orWhere('username', 'LIKE', '%' . $filterCreator . '%');
                });
            })
            ->when(isset($filterStatus), function ($query) use ($filterStatus) {
                if ($filterStatus != "all") {
                    return $query->where('is_verified', $filterStatus);
                }
            })
            ->when(!empty($request->filterDate) && $endDate == null, function ($query) use ($startDate) {
                $startDate = Carbon::createFromFormat('d/m/Y', trim($startDate));
                $query->whereDate('created_at', $startDate);
            })
            ->when(!empty($request->filterDate) && $endDate != null, function ($query) use ($startDate, $endDate) {
                $startDate = Carbon::createFromFormat('d/m/Y', trim($startDate));
                $endDate = Carbon::createFromFormat('d/m/Y', trim($endDate));
                $query->whereBetween('created_at', [$startDate, $endDate]);
            })
            ->when(!empty($request->search['value']), function ($query) use ($request) {
                $query->where(function ($subquery) use ($request) {
                    $subquery->where('from_name', 'LIKE', '%' . $request->search['value'] . '%')
                        ->orWhere('from_email', 'LIKE', '%' . $request->search['value'] . '%')
                        ->orWhere('reply_to_email', 'LIKE', '%' . $request->search['value'] . '%')
                        ->orWhere('company_address', 'LIKE', '%' . $request->search['value'] . '%')
                        ->orWhere('nickname', 'LIKE', '%' . $request->search['value'] . '%')
                        ->orWhereHas('user', function ($userQuery) use ($request) {
                            $userQuery->where('firstname', 'LIKE', '%' . $request->search['value'] . '%')
                                ->orWhere('lastname', 'LIKE', '%' . $request->search['value'] . '%')
                                ->orWhere('username', 'LIKE', '%' . $request->search['value'] . '%');
                        });
                });
            });


        return DataTables::of($senders)
            ->addColumn('checkbox', function ($item) {
                return '<input type="checkbox" id="chk-' . $item->id . '"
                                       class="form-check-input row-tic tic-check" name="check" value="' . $item->id . '"
                                       data-id="' . $item->id . '">';
            })
            ->addColumn('no', function () {
                static $counter = 0;
                $counter++;
                return $counter;
            })
            ->addColumn('senders', function ($item) {
                return ' <div></div><div><strong>' . $item->from_name . '</strong></div> <div><span
                       class="badge badge_4_XmD mantis_3oZCD">' . trans('From') . '</span> ' . $item->from_email . '</div> <div><span
                       class="badge badge_4_XmD slate_1qnpm">' . trans('Reply') . '</span> ' . $item->reply_to_email . '</div></div>';
            })
            ->addColumn('address', function ($item) {
                return $item->fullAddress();
            })
            ->addColumn('nickname', function ($item) {
                return $item->nickname;
            })
            ->addColumn('verified', function ($item) {
                if ($item->is_verified) {
                    return '<span> <i class="fal fa-check"></i></span>';
                } else {
                    return '<span><i class="fal fa-times"></i></span>';
                }
            })
            ->addColumn('creator', function ($item) {
                $userProfileRoute = route('admin.user.view.profile', $item->user_id);
                return '<a class="d-flex align-items-center me-2" href="' . $userProfileRoute . '">
                                <div class="flex-shrink-0">
                                  ' . optional($item->user)->profilePicture() . '
                                </div>
                                <div class="flex-grow-1 ms-3">
                                  <h5 class="text-hover-primary mb-0">' . optional($item->user)->firstname . ' ' . optional($item->user)->lastname . '</h5>
                                  <span class="fs-6 text-body">' . optional($item->user)->username . '</span>
                                </div>
                              </a>';
            })
            ->addColumn('created_at', function ($item) {
                return dateTime($item->created_at, basicControl()->date_time_format);
            })
            ->addColumn('action', function ($item) {
                $delete = route('admin.senderDelete', $item->id);
                return '<div class="btn-group" role="group">
                      <a href="javascript:void(0)" data-bs-target="#delete" data-bs-toggle="modal"
                       data-route="' . $delete . '" class="btn btn-white btn-sm delete_btn">
                        <i class="fal fa-trash me-1"></i> ' . trans("Delete") . '
                      </a>
                  </div>';
            })
            ->rawColumns(['checkbox', 'senders', 'address', 'nickname', 'verified', 'creator', 'created_at', 'action'])
            ->make(true);
    }

    public function senderDelete($id)
    {
        $sender = Sender::findOrFail($id);
        $sender->delete();
        return back()->with('success', 'Deleted Successfully');
    }

    public function senderMultipleDelete(Request $request)
    {
        if ($request->strIds == null) {
            session()->flash('error', 'You do not select User.');
            return response()->json(['error' => 1]);
        } else {
            Sender::whereIn('id', $request->strIds)->delete();
            session()->flash('success', 'Sender has been deleted successfully');
            return response()->json(['success' => 1]);
        }
    }
}
