<?php

namespace App\Http\Controllers\Admin\GeneratedHistory;

use App\Http\Controllers\Controller;
use App\Models\ApiClient;
use App\Models\UnsubscribeClient;
use App\Models\UnsubscribeGroup;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Str;
use Yajra\DataTables\Facades\DataTables;

class UnsubscriberController extends Controller
{
    public function globalUnsubscribes()
    {
        return view('admin.unsubscribe.globalUnsubs');
    }

    public function globalUnsubscribesSearch(Request $request)
    {
        $unsubscribers = UnsubscribeClient::with('user:id,firstname,lastname,username,image,image_driver')
            ->whereHas('user')
            ->when(!empty($request->search['value']), function ($query) use ($request) {
                $query->where(function ($subquery) use ($request) {
                    $subquery->where('email_address', 'LIKE', '%' . $request->search['value'] . '%')
                        ->orWhereHas('user', function ($userQuery) use ($request) {
                            $userQuery->where('firstname', 'LIKE', '%' . $request->search['value'] . '%')
                                ->orWhere('lastname', 'LIKE', '%' . $request->search['value'] . '%')
                                ->orWhere('username', 'LIKE', '%' . $request->search['value'] . '%');
                        });
                });
            })
            ->orderBy('id', 'DESC');


        return DataTables::of($unsubscribers)
            ->addColumn('checkbox', function ($item) {
                return '<input type="checkbox" id="chk-' . $item->id . '"
                                       class="form-check-input row-tic tic-check" name="check" value="' . $item->id . '"
                                       data-id="' . $item->id . '">';
            })
            ->addColumn('no', function () {
                static $counter = 0;
                $counter++;
                return $counter;
            })
            ->addColumn('email', function ($item) {
                return $item->email_address;
            })
            ->addColumn('creator', function ($item) {
                $userProfileRoute = route('admin.user.view.profile', $item->user_id);
                return '<a class="d-flex align-items-center me-2" href="' . $userProfileRoute . '">
                                <div class="flex-shrink-0">
                                  ' . optional($item->user)->profilePicture() . '
                                </div>
                                <div class="flex-grow-1 ms-3">
                                  <h5 class="text-hover-primary mb-0">' . optional($item->user)->firstname . ' ' . optional($item->user)->lastname . '</h5>
                                  <span class="fs-6 text-body">' . optional($item->user)->username . '</span>
                                </div>
                              </a>';
            })
            ->addColumn('created_at', function ($item) {
                return dateTime($item->created_at, basicControl()->date_time_format);
            })
            ->addColumn('action', function ($item) {
                $delete = route('admin.globalUnsubscribesDelete', $item->id);
                return '<div class="btn-group" role="group">
                      <a href="javascript:void(0)" class="btn btn-white btn-sm delete_btn" data-bs-target="#delete"
                           data-bs-toggle="modal" data-route="' . $delete . '">
                        <i class="fal fa-trash me-1"></i> ' . trans("Delete") . '
                      </a>
                  </div>';
            })
            ->rawColumns(['checkbox', 'email', 'creator', 'created_at', 'action'])
            ->make(true);
    }

    public function globalUnsubscribesDelete($id)
    {
        $unsubscribes = UnsubscribeClient::findOrFail($id);
        $unsubscribes->delete();
        return back()->with('success', 'Deleted Successfully');
    }

    public function unsubscribesGroup()
    {
        $data['groupRecord'] = UnsubscribeGroup::selectRaw('COUNT(id) AS totalGroup')
            ->selectRaw('COUNT(CASE WHEN is_display = 1 THEN id END) AS inPreGroup')
            ->selectRaw('(COUNT(CASE WHEN is_display = 1 THEN id END) / COUNT(id)) * 100 AS inPreGroupPercentage')
            ->selectRaw('COUNT(CASE WHEN is_display = 0 THEN id END) AS outPreGroup')
            ->selectRaw('(COUNT(CASE WHEN is_display = 0 THEN id END) / COUNT(id)) * 100 AS outPreGroupPercentage')
            ->selectRaw('COUNT(CASE WHEN DATE(created_at) = CURRENT_DATE THEN id END) AS todayGroup')
            ->selectRaw('(COUNT(CASE WHEN DATE(created_at) = CURRENT_DATE THEN id END) / COUNT(id)) * 100 AS todayGroupPercentage')
            ->selectRaw('COUNT(CASE WHEN YEARWEEK(created_at, 1) = YEARWEEK(CURDATE(), 1) THEN id END) AS thisWeekGroup')
            ->selectRaw('(COUNT(CASE WHEN YEARWEEK(created_at, 1) = YEARWEEK(CURDATE(), 1) THEN id END) / COUNT(id)) * 100 AS thisWeekGroupPercentage')
            ->get()
            ->toArray();

        return view('admin.unsubscribe.groupList', $data);
    }

    public function unsubscribesGroupSearch(Request $request)
    {
        $filterName = $request->name;
        $filterCreator = $request->creator;
        $filterStatus = $request->filterStatus;
        $filterDate = explode('-', $request->filterDate);
        $startDate = $filterDate[0];
        $endDate = isset($filterDate[1]) ? trim($filterDate[1]) : null;

        $groups = UnsubscribeGroup::with('user:id,firstname,lastname,username,image,image_driver')
            ->whereHas('user')
            ->withCount('UnsubscribeClients')
            ->orderBy('id', 'DESC')
            ->when(isset($filterName), function ($query) use ($filterName) {
                return $query->where('group_name', 'LIKE', '%' . $filterName . '%');
            })
            ->when(isset($filterCreator), function ($query) use ($filterCreator) {
                $query->whereHas('user', function ($qq) use ($filterCreator) {
                    $qq->where('firstname', 'LIKE', '%' . $filterCreator . '%')
                        ->orWhere('lastname', 'LIKE', '%' . $filterCreator . '%')
                        ->orWhere('username', 'LIKE', '%' . $filterCreator . '%');
                });
            })
            ->when(isset($filterStatus), function ($query) use ($filterStatus) {
                if ($filterStatus != "all") {
                    return $query->where('is_display', $filterStatus);
                }
            })
            ->when(!empty($request->filterDate) && $endDate == null, function ($query) use ($startDate) {
                $startDate = Carbon::createFromFormat('d/m/Y', trim($startDate));
                $query->whereDate('created_at', $startDate);
            })
            ->when(!empty($request->filterDate) && $endDate != null, function ($query) use ($startDate, $endDate) {
                $startDate = Carbon::createFromFormat('d/m/Y', trim($startDate));
                $endDate = Carbon::createFromFormat('d/m/Y', trim($endDate));
                $query->whereBetween('created_at', [$startDate, $endDate]);
            })
            ->when(!empty($request->search['value']), function ($query) use ($request) {
                $query->where(function ($subquery) use ($request) {
                    $subquery->where('group_name', 'LIKE', '%' . $request->search['value'] . '%')
                        ->orWhere('description', 'LIKE', '%' . $request->search['value'] . '%')
                        ->orWhereHas('user', function ($userQuery) use ($request) {
                            $userQuery->where('firstname', 'LIKE', '%' . $request->search['value'] . '%')
                                ->orWhere('lastname', 'LIKE', '%' . $request->search['value'] . '%')
                                ->orWhere('username', 'LIKE', '%' . $request->search['value'] . '%');
                        });
                });
            });


        return DataTables::of($groups)
            ->addColumn('no', function () {
                static $counter = 0;
                $counter++;
                return $counter;
            })
            ->addColumn('groupName', function ($item) {
                return $item->group_name;
            })
            ->addColumn('description', function ($item) {
                return Str::limit($item->description, 50);
            })
            ->addColumn('unsubscribes', function ($item) {
                return '<span class="badge bg-soft-dark text-dark">' . $item->unsubscribe_clients_count . '</span>';
            })
            ->addColumn('display', function ($item) {
                if ($item->is_display == 1) {
                    return '<span class="badge bg-soft-success text-success">
                    <span class="legend-indicator bg-success"></span>' . trans('In Preferences') . '
                  </span>';

                } else {
                    return '<span class="badge bg-soft-danger text-danger">
                    <span class="legend-indicator bg-danger"></span>' . trans('Out of Preferences') . '
                  </span>';
                }
            })
            ->addColumn('creator', function ($item) {
                $userProfileRoute = route('admin.user.view.profile', $item->user_id);
                return '<a class="d-flex align-items-center me-2" href="' . $userProfileRoute . '">
                                <div class="flex-shrink-0">
                                  ' . optional($item->user)->profilePicture() . '
                                </div>
                                <div class="flex-grow-1 ms-3">
                                  <h5 class="text-hover-primary mb-0">' . optional($item->user)->firstname . ' ' . optional($item->user)->lastname . '</h5>
                                  <span class="fs-6 text-body">' . optional($item->user)->username . '</span>
                                </div>
                              </a>';
            })
            ->addColumn('created_at', function ($item) {
                return dateTime($item->created_at, basicControl()->date_time_format);
            })
            ->rawColumns(['groupName', 'description', 'unsubscribes', 'display', 'creator', 'created_at'])
            ->make(true);
    }


    public function apiRecipient()
    {
        $data['recipientRecord'] = ApiClient::selectRaw('COUNT(id) AS totalRecipient')
            ->selectRaw('COUNT(CASE WHEN is_blocked = 0 THEN id END) AS activeRecipient')
            ->selectRaw('(COUNT(CASE WHEN is_blocked = 0 THEN id END) / COUNT(id)) * 100 AS activeRecipientPercentage')
            ->selectRaw('COUNT(CASE WHEN is_blocked = 1 THEN id END) AS blockedRecipient')
            ->selectRaw('(COUNT(CASE WHEN is_blocked = 1 THEN id END) / COUNT(id)) * 100 AS blockedRecipientPercentage')
            ->selectRaw('COUNT(CASE WHEN DATE(created_at) = CURRENT_DATE THEN id END) AS todayRecipient')
            ->selectRaw('(COUNT(CASE WHEN DATE(created_at) = CURRENT_DATE THEN id END) / COUNT(id)) * 100 AS todayRecipientPercentage')
            ->selectRaw('COUNT(CASE WHEN YEARWEEK(created_at, 1) = YEARWEEK(CURDATE(), 1) THEN id END) AS thisWeekRecipient')
            ->selectRaw('(COUNT(CASE WHEN YEARWEEK(created_at, 1) = YEARWEEK(CURDATE(), 1) THEN id END) / COUNT(id)) * 100 AS thisWeekRecipientPercentage')
            ->get()
            ->toArray();

        return view('admin.unsubscribe.apiRecipient', $data);
    }

    public function apiRecipientSearch(Request $request)
    {
        $filterIp = $request->ip;
        $filterCreator = $request->creator;
        $filterStatus = $request->filterStatus;
        $filterDate = explode('-', $request->filterDate);
        $startDate = $filterDate[0];
        $endDate = isset($filterDate[1]) ? trim($filterDate[1]) : null;

        $apiClients = ApiClient::with('user:id,firstname,lastname,username,image,image_driver')
            ->orderBy('id', 'DESC')
            ->when(isset($filterIp), function ($query) use ($filterIp) {
                return $query->where('ip', 'LIKE', '%' . $filterIp . '%');
            })
            ->when(isset($filterCreator), function ($query) use ($filterCreator) {
                $query->whereHas('user', function ($qq) use ($filterCreator) {
                    $qq->where('firstname', 'LIKE', '%' . $filterCreator . '%')
                        ->Orwhere('lastname', 'LIKE', '%' . $filterCreator . '%')
                        ->Orwhere('username', 'LIKE', '%' . $filterCreator . '%');
                });
            })
            ->when(isset($filterStatus), function ($query) use ($filterStatus) {
                if ($filterStatus != "all") {
                    return $query->where('is_blocked', $filterStatus);
                }
            })
            ->when(!empty($request->filterDate) && $endDate == null, function ($query) use ($startDate) {
                $startDate = Carbon::createFromFormat('d/m/Y', trim($startDate));
                $query->whereDate('created_at', $startDate);
            })
            ->when(!empty($request->filterDate) && $endDate != null, function ($query) use ($startDate, $endDate) {
                $startDate = Carbon::createFromFormat('d/m/Y', trim($startDate));
                $endDate = Carbon::createFromFormat('d/m/Y', trim($endDate));
                $query->whereBetween('created_at', [$startDate, $endDate]);
            })
            ->when(!empty($request->search['value']), function ($query) use ($request) {
                $query->where(function ($subquery) use ($request) {
                    $subquery->where('ip', 'LIKE', '%' . $request->search['value'] . '%')
                        ->orWhere('country_name', 'LIKE', '%' . $request->search['value'] . '%')
                        ->orWhere('country_code', 'LIKE', '%' . $request->search['value'] . '%')
                        ->orWhere('region_code', 'LIKE', '%' . $request->search['value'] . '%')
                        ->orWhere('region_name', 'LIKE', '%' . $request->search['value'] . '%')
                        ->orWhere('city_name', 'LIKE', '%' . $request->search['value'] . '%')
                        ->orWhere('zip_code', 'LIKE', '%' . $request->search['value'] . '%')
                        ->orWhere('latitude', 'LIKE', '%' . $request->search['value'] . '%')
                        ->orWhere('longitude', 'LIKE', '%' . $request->search['value'] . '%')
                        ->orWhereHas('user', function ($userQuery) use ($request) {
                            $userQuery->where('firstname', 'LIKE', '%' . $request->search['value'] . '%')
                                ->orWhere('lastname', 'LIKE', '%' . $request->search['value'] . '%')
                                ->orWhere('username', 'LIKE', '%' . $request->search['value'] . '%');
                        });
                });
            });


        return DataTables::of($apiClients)
            ->addColumn('no', function () {
                static $counter = 0;
                $counter++;
                return $counter;
            })
            ->addColumn('ip', function ($item) {
                return $item->ip;
            })
            ->addColumn('country', function ($item) {
                return $item->country_name;
            })
            ->addColumn('city', function ($item) {
                return $item->city_name;
            })
            ->addColumn('zip', function ($item) {
                return $item->zip_code;
            })
            ->addColumn('lanLong', function ($item) {
                return $item->latitude . ' - ' . $item->longitude;
            })
            ->addColumn('status', function ($item) {
                if ($item->is_blocked == 0) {
                    return '<span class="badge bg-soft-success text-success">
                    <span class="legend-indicator bg-success"></span>' . trans('Active') . '
                  </span>';

                } else {
                    return '<span class="badge bg-soft-danger text-danger">
                    <span class="legend-indicator bg-danger"></span>' . trans('Blocked') . '
                  </span>';
                }
            })
            ->addColumn('owner', function ($item) {
                $userProfileRoute = route('admin.user.view.profile', $item->user_id);
                return '<a class="d-flex align-items-center me-2" href="' . $userProfileRoute . '">
                                <div class="flex-shrink-0">
                                  ' . optional($item->user)->profilePicture() . '
                                </div>
                                <div class="flex-grow-1 ms-3">
                                  <h5 class="text-hover-primary mb-0">' . optional($item->user)->firstname . ' ' . optional($item->user)->lastname . '</h5>
                                  <span class="fs-6 text-body">' . optional($item->user)->username . '</span>
                                </div>
                              </a>';
            })
            ->addColumn('created_at', function ($item) {
                return dateTime($item->created_at, basicControl()->date_time_format);
            })
            ->rawColumns(['ip', 'country', 'city', 'zip', 'lanLong', 'status', 'owner', 'created_at'])
            ->make(true);
    }

    public function globalUnsubscribesMultiDelete(Request $request)
    {
        if ($request->strIds == null) {
            session()->flash('error', 'You do not select User.');
            return response()->json(['error' => 1]);
        } else {
            UnsubscribeClient::whereIn('id', $request->strIds)->delete();
            session()->flash('success', 'Global Unsubscribes has been deleted successfully');
            return response()->json(['success' => 1]);
        }
    }
}
