<?php

namespace App\Http\Controllers;

use App\Jobs\SingleSendEmails;
use App\Mail\SendMail;
use App\Models\ContentDetails;
use App\Models\Deposit;
use App\Models\Gateway;
use App\Models\Language;
use App\Models\NotificationTemplate;
use App\Models\PageDetail;
use App\Models\Plan;
use App\Models\Sender;
use App\Models\SingleSend;
use App\Models\Subscribe;
use App\Models\User;
use App\Services\CheckEmailAddress;
use App\Traits\Frontend;
use App\Traits\Notify;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Validator;
use Stevebauman\Purify\Facades\Purify;
use Facades\App\Services\BasicService;

class FrontendController extends Controller
{
    use Notify, Frontend;

    public function __construct()
    {
        $this->theme = template();
    }


    public function page($slug = '/')
    {
        $selectedTheme = basicControl()->theme;
        $existingSlugs = collect([]);
        DB::table('pages')->select('slug')->get()->map(function ($item) use ($existingSlugs) {
            $existingSlugs->push($item->slug);
        });
        if (!in_array($slug, $existingSlugs->toArray())) {
            abort(404);
        }

        try {
            $pageDetails = PageDetail::with('page')
                ->whereHas('page', function ($query) use ($slug, $selectedTheme) {
                    $query->where(['slug' => $slug, 'template_name' => $selectedTheme]);
                })
                ->first();


            $pageSeo = [
                'page_title' => optional($pageDetails->page)->page_title,
                'meta_title' => optional($pageDetails->page)->meta_title,
                'meta_keywords' => implode(',', optional($pageDetails->page)->meta_keywords ?? []),
                'meta_description' => optional($pageDetails->page)->meta_description,
                'og_description' => optional($pageDetails->page)->og_description,
                'meta_robots' => optional($pageDetails->page)->meta_robots,
                'meta_image' => getFile(optional($pageDetails->page)->meta_image_driver, optional($pageDetails->page)->meta_image),
                'breadcrumb_image' => optional($pageDetails->page)->breadcrumb_status ?
                    getFile(optional($pageDetails->page)->breadcrumb_image_driver, optional($pageDetails->page)->breadcrumb_image) : null,
            ];

            $sectionsData = $this->getSectionsData($pageDetails->sections, $pageDetails->content, $selectedTheme);
            return view("themes.{$selectedTheme}.page", compact('sectionsData', 'pageSeo'));

        } catch (\Exception $e) {
            return redirect()->route('instructionPage');
        }
    }

    public function blogDetails(Request $request)
    {
        $search = $request->all();
        $data['blogDetails'] = ContentDetails::select(['id', 'description', 'content_id'])->with('content')
            ->where('id', $request->id)->firstOrFail();

        $data['popularContentDetails'] = ContentDetails::select('id', 'content_id', 'description', 'created_at')
            ->where('id', '!=', $request->id)
            ->whereHas('content', function ($query) {
                return $query->where('type', 'multiple')->whereIn('name', ['blog']);
            })
            ->when(isset($search['title']), function ($query) use ($search) {
                $query->where('description', 'LIKE', '%' . $search['title'] . '%');
            })
            ->get()->groupBy('content.name');

        return view($this->theme . 'blog_details', $data);
    }

    public function senderVerifyMail($utr)
    {
        $sender = Sender::where('utr', $utr)
            ->where('is_verified', 0)
            ->where('send_at', '>', Carbon::now()->subHours(48))
            ->firstOrFail();

        $sender->is_verified = 1;
        $sender->save();

        return view($this->theme . 'user.senders.verified');
    }

    public function subscribe(Request $request)
    {
        $purifiedData = Purify::clean($request->all());
        $validationRules = [
            'email' => 'required|email|min:8|max:100|unique:subscribes',
        ];
        $validate = Validator::make($purifiedData, $validationRules);
        if ($validate->fails()) {
            session()->flash('error', 'Email Field is required');
            return back()->withErrors($validate)->withInput();
        }
        $purifiedData = (object)$purifiedData;

        $subscribe = new Subscribe();
        $subscribe->email = $purifiedData->email;
        $subscribe->save();

        return back()->with('success', 'Subscribed successfully');
    }

    public function contactSend(Request $request)
    {
        $this->validate($request, [
            'name' => 'required|max:50',
            'email' => 'required|email|max:91',
            'subject' => 'required|max:100',
            'message' => 'required|max:1000',
        ]);
        $requestData = Purify::clean($request->except('_token', '_method'));

        $name = $requestData['name'];
        $email_from = $requestData['email'];
        $subject = $requestData['subject'];
        $message = $requestData['message'] . "<br>Regards<br>" . $name;
        $from = $email_from;

        Mail::to(basicControl()->sender_email)->send(new SendMail($from, $subject, $message));
        return back()->with('success', 'Mail has been sent');
    }
}
