<?php

namespace App\Http\Controllers\User;

use App\Exports\ExportAutomationActivityEmail;
use App\Exports\ExportScheduleEmail;
use App\Http\Controllers\Controller;
use App\Http\Requests\AutomationStoreRequest;
use App\Models\Automation;
use App\Models\AutomationSchedule;
use App\Models\ContactType;
use App\Models\CustomField;
use App\Models\EmailAutomationMap;
use App\Models\EmailTemplate;
use App\Models\Segment;
use App\Models\Sender;
use App\Models\UnsubscribeGroup;
use App\Traits\Activity;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Maatwebsite\Excel\Facades\Excel;
use Ramsey\Uuid\Uuid;
use DateTime;

class AutomationController extends Controller
{
    use Activity;

    public function __construct()
    {
        $this->middleware(['auth']);
        $this->middleware(function ($request, $next) {
            $this->user = auth()->user();
            return $next($request);
        });
        $this->theme = template();
    }

    public function index(Request $request)
    {
        $search = $request->all();
        $dateSearch = $request->live_at;
        $date = preg_match("/^[0-9]{2,4}\-[0-9]{1,2}\-[0-9]{1,2}$/", $dateSearch);

        $data['automations'] = Automation::own()->withCount(['automationSchedule'])->orderBy('id', 'desc')
            ->when(isset($search['automation_name']), function ($query) use ($search) {
                return $query->where('automation_name', 'LIKE', '%' . $search['automation_name'] . '%');
            })
            ->when($date == 1, function ($query) use ($dateSearch) {
                return $query->whereDate("live_at", $dateSearch);
            })
            ->when(isset($search['status']) && $search['status'] != null, function ($query) use ($search) {
                if ($search['status'] == 'live') {
                    return $query->where('status', 1);
                } elseif ($search['status'] == 'draft') {
                    return $query->where('status', 0);
                }
            })
            ->paginate(basicControl()->paginate);
        return view($this->theme . 'user.automation.index', $data);
    }

    public function choose()
    {
        return view($this->theme . 'user.automation.automation_type');
    }

    public function create($type)
    {
        if ($type == 'custom') {
            $automation = new Automation();
            $automation->user_id = $this->user->id;
            $automation->automation_name = 'Untitled Automation';
            $automation->automation_type = 'custom';
            $automation->utr = (Uuid::uuid1())->toString();
            $automation->status = 0;
            $automation->save();

            $sender = Sender::own()->where('is_verified', 1)->orderBy('id', 'desc')->first();

            AutomationSchedule::create([
                'user_id' => $automation->user_id,
                'automation_id' => $automation->id,
                'sender_id' => $sender->id ?? null,
                'subject' => 'Welcome' ?? null,
                'waiting_type' => 'instantly',
                'utr' => (Uuid::uuid1())->toString(),
            ]);

            $route = route('user.automationShow', $automation->utr);
            $this->userActivity(
                "A draft automation <a href='$route' class='text-primary'>$automation->automation_name</a> was created!"
            );

            return redirect()->route('user.automationShow', $automation->utr);
        }
    }

    public function show($utr)
    {
        $data['automation'] = Automation::own()->with(['automationSchedule:id,user_id,automation_id,sender_id,subject,waiting_period,waiting_type,utr,template_id,preview_img'])
            ->where('utr', $utr)->firstOrFail();

        $data['contactTypes'] = ContactType::own(1)->get();
        $data['segments'] = Segment::own()->get();
        $data['unsubscribeGroups'] = UnsubscribeGroup::own()->get();
        $data['senders'] = Sender::own()->where('is_verified', 1)->get();
        return view($this->theme . 'user.automation.create', $data);
    }

    public function scheduleUpdate(Request $request)
    {
        $currentSchedule = AutomationSchedule::own()->find($request->scheduleId);

        $previousSchedule = AutomationSchedule::own()
            ->where('id', '<', $currentSchedule->id)
            ->where('automation_id', $currentSchedule->automation_id)
            ->orderBy('id', 'desc')
            ->first();

        $currentSchedule->waiting_period = $request->waiting_period;
        if ($request->email_type == 'instant' && !$previousSchedule) {
            $currentSchedule->waiting_type = 'instantly';
        } else {
            $currentSchedule->waiting_type = $request->waiting_type;
        }
        $currentSchedule->save();
        $schedules = AutomationSchedule::own()->select(['id', 'user_id', 'automation_id', 'sender_id', 'subject',
            'waiting_period', 'waiting_type', 'utr', 'template_id', 'preview_img'])
            ->where('automation_id', $currentSchedule->automation_id)->get();

        $this->userActivity("A automation schedule was updated!");
        return response()->json(['status' => 'success', 'data' => $schedules]);
    }

    public function scheduleAdd(Request $request)
    {
        $automation = Automation::own()->find($request->automationId);
        $sender = Sender::own()->where('is_verified', 1)->orderBy('id', 'desc')->first();

        if ($automation) {
            AutomationSchedule::create([
                'user_id' => $this->user->id,
                'automation_id' => $automation->id,
                'sender_id' => $sender->id ?? null,
                'subject' => 'Welcome' ?? null,
                'waiting_period' => 2,
                'waiting_type' => 'days',
                'utr' => (Uuid::uuid1())->toString(),
            ]);

            $schedules = AutomationSchedule::own()->select(['id', 'user_id', 'automation_id', 'sender_id', 'subject',
                'waiting_period', 'waiting_type', 'utr', 'template_id', 'preview_img'])
                ->where('automation_id', $automation->id)->get();

            $this->userActivity("A automation schedule was added!");
            return response()->json(['status' => 'success', 'data' => $schedules]);
        }
    }

    public function scheduleDelete(Request $request)
    {
        $currentSchedule = AutomationSchedule::own()->find($request->scheduleId);
        $currentSchedule->delete();
        $schedules = AutomationSchedule::own()->select(['id', 'user_id', 'automation_id', 'sender_id', 'subject',
            'waiting_period', 'waiting_type', 'utr', 'template_id', 'preview_img'])
            ->where('automation_id', $request->automationId)->get();

        $this->userActivity("A new automation schedule was deleted!");
        return response()->json(['status' => 'success', 'data' => $schedules]);
    }

    public function scheduleSender(Request $request)
    {
        $currentSchedule = AutomationSchedule::own()->find($request->activeScheduleIdSender);
        $currentSchedule->sender_id = $request->senderId;
        $currentSchedule->subject = $request->subject;
        $currentSchedule->save();
        $schedules = AutomationSchedule::own()->select(['id', 'user_id', 'automation_id', 'sender_id', 'subject',
            'waiting_period', 'waiting_type', 'utr', 'template_id', 'preview_img'])
            ->where('automation_id', $request->automationId)->get();

        $this->userActivity("A automation schedule sender and subject updated!");
        return response()->json(['status' => 'success', 'data' => $schedules]);
    }

    public function scheduleEmailTemplate($utr)
    {
        $data['templates'] = EmailTemplate::select(['id', 'user_id', 'template_name', 'preview_image', 'driver', 'custom'])
            ->where('user_id', null)->orWhere('user_id', auth()->id())->get();
        $data['utr'] = $utr;
        return view($this->theme . 'user.automation.show_template', $data);
    }

    public function scheduleEmailTemplatePost($templateId, $utr)
    {
        $template = EmailTemplate::findOrFail($templateId);
        $automationSchedule = AutomationSchedule::own()->where('utr', $utr)->firstOrFail();
        $automationSchedule->template_id = $template->id;
        $automationSchedule->preview_img = $template->preview_image;
        $automationSchedule->template_json = $template->json_code;
        $automationSchedule->template_html = $template->html_code;
        $automationSchedule->save();

        return redirect()->route('user.automationScheduleEmailDesign', $utr);
    }

    public function scheduleEmailDesign($utr)
    {
        $data['automationSchedule'] = AutomationSchedule::own()->with(['automation:id,utr', 'template'])->where('utr', $utr)->firstOrFail();
        $data['fields'] = CustomField::own()->orwhere('is_reserved', 1)->get();
        $data['automationUtr'] = $data['automationSchedule']->automation->utr ?? null;
        $data['isClassicEditor'] = ($data['automationSchedule']->template_id == 1) ? false : ($data['automationSchedule']->template_json === "{}");
        return view($this->theme . 'user.automation.editor', $data);
    }

    public function scheduleEmailDesignSave(Request $request, $utr)
    {
        $automationSchedule = AutomationSchedule::own()->where('utr', $utr)->first();
        $reqData = json_decode($request->getContent(), true);
        if ($automationSchedule) {
            $automationSchedule->template_json = $reqData['json_design'] ?? '{}';
            $automationSchedule->template_html = $reqData['html_design'];
            $automationSchedule->save();
        }
        return response()->json(['status' => 'success']);
    }

    public function automationSave(AutomationStoreRequest $request, $utr)
    {
        $automation = Automation::own()->where('utr', $utr)->firstOrFail();
        $getData = getIdAndTable($request->automation_for);
        $automation->automation_for = $getData['id'];
        $automation->automation_for_table = $getData['table'];
        $automation->automation_name = $request->automation_name;
        $automation->unsubscribe_group_id = $request->unsubscribe_group_id;
        $automation->save();

        $message = 'We have finished processing your request to create automation';
        $this->emailNotification($message, route('user.automation'), 'View your automation');

        $route = route('user.automationShow', $automation->utr);
        $this->userActivity(
            "You have finished processing to create <a href='$route' class='text-primary'>$automation->automation_name</a> automation!"
        );
        return redirect()->route('user.automation');
    }

    public function automationSetLive(AutomationStoreRequest $request, $utr)
    {
        $automation = Automation::own()->with(['automationSchedule'])->where('utr', $utr)->firstOrFail();
        if (count($automation->automationSchedule) > 0) {
            foreach ($automation->automationSchedule as $schedule) {
                if (!$schedule->template_html) {
                    return back()->withInput()->with('error', 'Please select email format for all emails');
                }
            }
        } else {
            return back()->withInput()->with('error', 'Please set email first for automation live');
        }

        $getData = getIdAndTable($request->automation_for);
        $automation->automation_for = $getData['id'];
        $automation->automation_for_table = $getData['table'];
        $automation->automation_name = $request->automation_name;
        $automation->unsubscribe_group_id = $request->unsubscribe_group_id;
        $automation->status = 1;
        $automation->live_at = Carbon::now();
        $automation->save();

        $message = 'We have finished processing your request to create automation and set live';
        $this->emailNotification($message, route('user.automation'), 'View your automation');

        $route = route('user.automationShow', $automation->utr);
        $this->userActivity(
            "You have finished processing to create <a href='$route' class='text-primary'>$automation->automation_name</a> automation and set live!"
        );

        return redirect()->route('user.automation');
    }

    public function automationDelete($utr)
    {
        Automation::own()->where('utr', $utr)->firstOrfail()->delete();
        $message = 'We have finished processing your request to delete automation';
        $this->emailNotification($message, route('user.automation'), 'View your automation');

        $this->userActivity(
            "You have finished processing to delete automation!"
        );

        return back()->with('success', 'Deleted Successfully');
    }

    public function automationBulkDelete(Request $request)
    {
        if ($request->strIds == null) {
            session()->flash('error', 'You do not select ID.');
            return response()->json(['error' => 1]);
        } else {
            Automation::own()->whereIn('id', $request->strIds)->get()->map(function ($query) {
                $query->delete();
            });
            $message = 'We have finished processing your request to delete multiple automation';
            $this->emailNotification($message, route('user.automation'), 'View your automation');

            $this->userActivity(
                "You have finished processing to delete multiple automation!"
            );
            session()->flash('success', 'Deleted Successfully');
            return response()->json(['status' => 'success']);
        }
    }

    public function automationDuplicate($utr)
    {
        $automation = Automation::own()->with(['automationSchedule'])->where('utr', $utr)->firstOrFail();
        try {
            $fillData = new Automation($automation->attributesToArray());
            $fillData['utr'] = (Uuid::uuid1())->toString();
            $fillData['status'] = 0;
            $fillData['live_at'] = null;
            $fillData['total_triggered'] = 0;
            $fillData['total_delivered'] = 0;
            $fillData['total_unique_opens'] = 0;
            $fillData['total_opens'] = 0;
            $fillData['total_bounces'] = 0;
            $fillData['total_unsubscribe'] = 0;
            $fillData['created_at'] = Carbon::now();
            $fillData['updated_at'] = Carbon::now();
            $fillData->save();

            if (count($automation->automationSchedule) > 0) {
                foreach ($automation->automationSchedule as $schedule) {
                    $scheduleData = new AutomationSchedule($schedule->attributesToArray());
                    $scheduleData['automation_id'] = $fillData->id;
                    $scheduleData->save();
                }
            }

            $message = 'We have finished processing your request to create duplicate automation';
            $this->emailNotification($message, route('user.automation'), 'View your automation');

            $route = route('user.automationShow', $fillData->utr);
            $this->userActivity(
                "You have finished processing to create duplicate <a href='$route' class='text-primary'>$automation->automation_name</a> automation!"
            );

            return back()->with('success', 'Automation Duplicated Successfully');
        } catch (\Exception $e) {
            return back()->with('error', $e->getMessage());
        }
    }

    public function automationPause($utr)
    {
        $automation = Automation::own()->where('utr', $utr)->where('status', 1)->firstOrFail();
        try {
            $automation->status = 0;
            $automation->save();
            $message = 'We have finished processing your request to paused automation';
            $this->emailNotification($message, route('user.automation'), 'View your automation');

            $route = route('user.automationShow', $automation->utr);
            $this->userActivity(
                "You have finished processing to paused <a href='$route' class='text-primary'>$automation->automation_name</a> automation!"
            );

            return back()->with('success', 'Automation Paused Successfully');
        } catch (\Exception $e) {
            return back()->with('alert', 'something went wrong.Please try again');
        }
    }

    public function automationStatics($utr)
    {
        $automation = Automation::own()->with(['automationSchedule', 'emailAutomationMaps', 'unsubscribeGroup',
            'contactType', 'segment'])
            ->where('utr', $utr)->firstOrFail();


        $data['deliveredPercent'] = $automation->getPercent()['deliveredPercent'];
        $data['bouncesPercent'] = $automation->getPercent()['bouncesPercent'];
        $data['uniqueOpenPercent'] = $automation->getPercent()['uniqueOpenPercent'];
        $data['unsubscribesPercent'] = $automation->getPercent()['unsubscribesPercent'];


        $data['horizontalBarChat'] = [$automation->total_triggered, $automation->total_delivered, $automation->total_bounces,
            $automation->total_unique_opens, $automation->total_opens, $automation->total_unsubscribe];

        $data['lineChartUniqueOpen'] = DB::table('email_automation_maps')
            ->select(DB::raw('open_date AS x'), DB::raw('COUNT(*) AS y'))
            ->where('automation_id', $automation->id)
            ->where('unique_opens', 1)
            ->where('user_id', $this->user->id)
            ->groupBy('open_date')
            ->get()
            ->toArray();


        $start = Carbon::now()->firstOfMonth();
        $end = Carbon::now()->lastOfMonth();

        $transactions = EmailAutomationMap::select(DB::raw("DATE_FORMAT(open_date, '%j') as day"))
            ->whereBetween('open_date', [$start, $end])
            ->selectRaw("COUNT(CASE WHEN unique_opens = '1' THEN 1 END) as `Unique`")
            ->selectRaw("COUNT(CASE WHEN unsubscribes = '1' THEN 1 END) as `Unsubscribe`")
            ->selectRaw("open_date as `open_date`")
            ->where('user_id', $this->user->id)
            ->where('automation_id', $automation->id)
            ->groupBy('day')
            ->get();

        $labels = [];
        $dataUnique = [];
        $dataUnsubscribe = [];
        $start = new DateTime($start);
        $end = new DateTime($end);

        for ($day = $start; $day <= $end; $day->modify('+1 day')) {
            $i = $day->format('j');
            $labels[] = $day->format('jS M');
            $currentUnique = 0;
            $currentUnsubscribe = 0;

            if (isset($transactions)) {
                foreach ($transactions as $key => $transaction) {
                    if (Carbon::parse($transaction->open_date) == $day) {
                        $currentUnique = $transaction->Unique;
                        $currentUnsubscribe = $transaction->Unsubscribe;
                        break;
                    }
                }
            }

            $dataUnique[] = $currentUnique;
            $dataUnsubscribe[] = $currentUnsubscribe;
        }

        $data['labels'] = $labels;
        $data['dataUnique'] = $dataUnique;
        $data['dataUnsubscribe'] = $dataUnsubscribe;
        $data['automation'] = $automation;
        return view($this->theme . 'user.automation.stats', $data);
    }

    public function automationsGetChart(Request $request, $utr)
    {
        $automation = Automation::own()->select(['id', 'utr'])->where('utr', $utr)->first();
        $start = $request->start;
        $end = $request->end;

        $transactions = EmailAutomationMap::select(DB::raw("DATE_FORMAT(open_date, '%j') as day"))
            ->whereBetween('open_date', [$start, $end])
            ->selectRaw("COUNT(CASE WHEN unique_opens = '1' THEN 1 END) as `Unique`")
            ->selectRaw("COUNT(CASE WHEN unsubscribes = '1' THEN 1 END) as `Unsubscribe`")
            ->selectRaw("open_date as `open_date`")
            ->where('user_id', $this->user->id)
            ->where('automation_id', $automation->id)
            ->groupBy('day')
            ->get();

        $labels = [];
        $dataUnique = [];
        $dataUnsubscribe = [];
        $start = new DateTime($start);
        $end = new DateTime($end);

        for ($day = $start; $day <= $end; $day->modify('+1 day')) {
            $i = $day->format('j');
            $labels[] = $day->format('jS M');
            $currentUnique = 0;
            $currentUnsubscribe = 0;

            if (isset($transactions)) {
                foreach ($transactions as $key => $transaction) {
                    if (Carbon::parse($transaction->open_date) == $day) {
                        $currentUnique = $transaction->Unique;
                        $currentUnsubscribe = $transaction->Unsubscribe;
                        break;
                    }
                }
            }

            $dataUnique[] = $currentUnique;
            $dataUnsubscribe[] = $currentUnsubscribe;
        }

        $data['labels'] = $labels;
        $data['dataUnique'] = $dataUnique;
        $data['dataUnsubscribe'] = $dataUnsubscribe;

        return response()->json($data);
    }

    public function automationsLogs(Request $request, $utr)
    {
        $search = $request->all();
        $dateSearch = $request->created_at;
        $date = preg_match("/^[0-9]{2,4}\-[0-9]{1,2}\-[0-9]{1,2}$/", $dateSearch);

        $automation = Automation::own()->select(['id', 'utr', 'automation_name'])->latest()->where('utr', $utr)->firstOrFail();
        $data['recipients'] = EmailAutomationMap::own()->where('automation_id', $automation->id)
            ->when(isset($request->logs), function ($query) use ($request) {
                if ($request->logs == 'delivered') {
                    return $query->where('delivered', 1);
                } elseif ($request->logs == 'bounces') {
                    return $query->where('bounces', 1);
                } elseif ($request->logs == 'unique_opens') {
                    return $query->where('unique_opens', 1);
                } elseif ($request->logs == 'unsubscribe') {
                    return $query->where('unsubscribes', 1);
                }
            })
            ->when(isset($search['email']), function ($query) use ($search) {
                return $query->where('email_address', 'LIKE', '%' . $search['email'] . '%');
            })
            ->when($date == 1, function ($query) use ($dateSearch) {
                return $query->whereDate("created_at", $dateSearch);
            })
            ->orderBy('id', 'desc')
            ->paginate(basicControl()->paginate);

        $data['logsType'] = $request->logs;
        $data['title'] = ucfirst(str_replace("_", " ", $request->logs));
        return view($this->theme . 'user.automation.log.index', $data, compact('automation'));
    }

    public function recipientExportCsv($automationId, $type)
    {
        Automation::own()->select(['id'])->findOrFail($automationId);
        $this->userActivity("You have finished processing to export recipients!");
        return Excel::download(new ExportAutomationActivityEmail($automationId, $type), 'emails.csv');
    }

    public function automationsScheduleLogs($utr)
    {
        $schedule = AutomationSchedule::own()->select(['id', 'utr'])->where('utr', $utr)->firstOrFail();
        $data['recipients'] = EmailAutomationMap::own()->where('automation_schedule_id', $schedule->id)
            ->paginate(basicControl()->paginate);
        $data['schedule'] = $schedule;
        return view($this->theme . 'user.automation.log.schedule-index', $data);
    }

    public function ScheduleRecipientExportCsv($scheduleId)
    {
        AutomationSchedule::own()->select(['id', 'utr'])->findOrFail($scheduleId);
        $this->userActivity("You have finished processing to export recipients!");
        return Excel::download(new ExportScheduleEmail($scheduleId), 'emails.csv');
    }
}
