<?php

namespace App\Http\Controllers\User;

use App\Exports\BulkEmailResultExport;
use App\Http\Controllers\Controller;
use App\Models\BulkEmailCheck;
use App\Traits\Activity;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Facades\Tintnaingwin\EmailChecker\EmailChecker;
use Illuminate\Support\Facades\DB;
use Maatwebsite\Excel\Facades\Excel;
use  Facades\App\Console\Commands\BulkEmailCheckDelete;

class BulkEmailValidation extends Controller
{
    use Activity;

    public function __construct()
    {
        $this->middleware(['auth']);
        $this->middleware(function ($request, $next) {
            $this->user = auth()->user();
            return $next($request);
        });
        $this->theme = template();
    }

    public function bulkEmailValidation(Request $request)
    {
        if ($request->method() == 'GET') {
            $basic = basicControl();
            return view($this->theme . 'user.bulk-validation.index', compact('basic'));
        } elseif ($request->method() == 'POST') {
            try {
                if (auth()->user()->credits <= 0) {
                    return back()->with('error', 'Your account balance is currently at zero credits. To proceed, please acquire prepaid credits through our pricing options.')->withInput();
                }
                $emailArray = [];
                if ($request->emailAddress) {
                    $emailArray = explode(',', $request->emailAddress);

                } elseif ($request->has('file')) {
                    if ($request->file->getClientOriginalExtension() != 'csv') {
                        return back()->with('error', 'Only accepted .csv files');
                    }
                    $file = fopen($request->file->getRealPath(), 'r');
                    $firstIteration = true;
                    while ($csvLine = fgetcsv($file)) {
                        if ($firstIteration) {
                            $firstIteration = false;
                            continue; // Skip the first iteration
                        }
                        if (isset($csvLine[0])) {
                            $emailArray[] = $csvLine[0];
                        }
                    }
                } else {
                    return back()->with('error', 'Email Addresses or CSV file is required');
                }

                $utr = $this->checkEmailAddress($emailArray);
                return redirect()->route('user.bulkEmailResult', $utr);
            } catch (\Exception $e) {
                return back()->with('error', $e->getMessage());
            }
        }
    }

    public function checkEmailAddress($emailArr = []): string
    {
        set_time_limit(300);
        $basic = basicControl();
        $count = 1;
        $countEmailArr = count($emailArr);
        $maxLimit = $basic->email_check_max_limit;
        $userId = auth()->id();
        $user = auth()->user();
        $utr = strRandom(10);

        if ($countEmailArr > 0) {
            $insertData = [];

            foreach ($emailArr as $email) {
                if ($maxLimit < $count || $user->credits < $count) {
                    break;
                }

                $status = filter_var($email, FILTER_VALIDATE_EMAIL) ? EmailChecker::check($email) : false;

                $insertData[] = [
                    'user_id' => $userId,
                    'utr' => $utr,
                    'email' => $email,
                    'status' => $status ? 'VALID' : 'INVALID',
                    'created_at' => now(),
                    'updated_at' => now(),
                ];

                $count++;
            }

            // Insert data in batches (you can adjust the batch size as needed)
            $batchSize = 500;
            $chunks = array_chunk($insertData, $batchSize);

            foreach ($chunks as $chunk) {
                DB::table('bulk_email_checks')->insert($chunk);
            }

            $user->credits -= $count;
            $user->save();
        }
        return $utr;
    }

    public function bulkEmailResult($utr)
    {
        $data['emailLists'] = BulkEmailCheck::where('user_id', auth()->id())->where('utr', $utr)->get();
        $data['utr'] = $data['emailLists'][0]->utr ?? null;
        return view($this->theme . 'user.bulk-validation.result', $data);
    }

    public function bulkEmailExport(Request $request)
    {
        $this->userActivity(
            "You have finished processing to export bulk email checker result!"
        );
        return Excel::download(new BulkEmailResultExport($request->type, $request->utr), 'emails.csv');
    }

    public function bulkEmailCsvSample()
    {
        $file = 'bulk-email-sample.csv';
        $full_path = 'assets/' . $file;
        $title = 'bulk-email-sample';
        $ext = pathinfo($file, PATHINFO_EXTENSION);
        $mimetype = mime_content_type($full_path);
        header('Content-Disposition: attachment; filename="' . $title . '.' . $ext . '";');
        header("Content-Type: " . $mimetype);
        return readfile($full_path);
    }
}
