<?php

namespace App\Http\Controllers\User;

use App\Exports\ContactExport;
use App\Exports\ExportInvalidEmails;
use App\Http\Controllers\Controller;
use App\Http\Requests\ContactTypeRequest;
use App\Models\Contact;
use App\Models\ContactType;
use App\Models\Country;
use App\Models\CustomField;
use App\Models\SingleSend;
use App\Traits\Activity;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Maatwebsite\Excel\Facades\Excel;
use App\Mail\SingleSendMail;

class ContactController extends Controller
{
    use Activity;

    public function __construct()
    {
        $this->middleware(['auth']);
        $this->middleware(function ($request, $next) {
            $this->user = auth()->user();
            return $next($request);
        });
        $this->theme = template();
    }

    public function contactTypeList(Request $request)
    {
        $search = $request->all();
        $data['contactTypes'] = ContactType::own(1)->withCount('contacts')
            ->when(isset($search['name']), function ($query) use ($search) {
                $query->where('name', 'LIKE', '%' . $search['name'] . '%');
            })
            ->orderByRaw('ISNULL(created_at) DESC, created_at ASC')
            ->paginate(basicControl()->paginate);
        return view($this->theme . 'user.contacts.contactTypeIndex', $data);
    }

    public function contactTypeSave(ContactTypeRequest $request)
    {
        $contactType = new ContactType();
        $fillData = $request->only($contactType->getFillable());
        $fillData['user_id'] = $this->user->id;
        $contactType->fill($fillData)->save();

        $this->userActivity("You have finished processing to create contact type!");

        session()->flash('success', 'Added Successfully');
        return response()->json(['status' => 'success']);
    }

    public function contactTypeUpdate(ContactTypeRequest $request)
    {
        $contactType = ContactType::own()->find($request->id);
        if ($contactType) {
            $fillData = $request->only($contactType->getFillable());
            $contactType->fill($fillData)->save();

            $this->userActivity("You have finished processing to update contact type!");

            session()->flash('success', 'Updated Successfully');
            return response()->json(['status' => 'success']);
        }
    }

    public function contactTypeDelete($id)
    {
        $contactType = ContactType::own()->findOrFail($id);
        $contactType->delete();
        $message = 'We have finished processing your request to delete contact type';
        $this->emailNotification($message, route('user.contactTypeList'), 'View your contact type');

        $this->userActivity(
            "You have finished processing to delete contact type!"
        );

        return back()->with('success', 'Deleted Successfully');
    }

    public function contactTypeBulkDelete(Request $request)
    {
        if ($request->strIds == null) {
            session()->flash('error', 'You do not select ID.');
            return response()->json(['error' => 1]);
        } else {
            ContactType::own()->whereIn('id', $request->strIds)->get()->map(function ($query) {
                $query->delete();
            });
            $message = 'We have finished processing your request to delete multiple contact type';
            $this->emailNotification($message, route('user.contactTypeList'), 'View your contact type');

            $this->userActivity(
                "You have finished processing to delete multiple contact type!"
            );

            session()->flash('success', 'Deleted Successfully');
            return response()->json(['status' => 'success']);
        }
    }

    public function contactList(Request $request, $type_id)
    {
        $search = $request->all();
        $data['contactType'] = ContactType::own(1)->select(['id', 'name'])->findOrFail($type_id);
        $data['contacts'] = Contact::own()->with('contactType')
            ->when(isset($search['first_name']), function ($query) use ($search) {
                $query->where('first_name', 'LIKE', '%' . $search['first_name'] . '%');
            })
            ->when(isset($search['last_name']), function ($query) use ($search) {
                $query->where('last_name', 'LIKE', '%' . $search['last_name'] . '%');
            })
            ->when(isset($search['email']), function ($query) use ($search) {
                $query->where('email', 'LIKE', '%' . $search['email'] . '%');
            })
            ->when($type_id != 1, function ($query) use ($search, $type_id) {
                $query->where('contact_type_id', $type_id);
            })
            ->orderBy('id', 'desc')
            ->paginate(basicControl()->paginate);
        return view($this->theme . 'user.contacts.contactList.index', $data);
    }

    public function contactAdd(Request $request)
    {
        $data['customFields'] = CustomField::own()->get();
        if ($request->method() == 'GET') {
            $data['countries'] = Country::get();
            $data['contactTypes'] = ContactType::own(1)->get();
            return view($this->theme . 'user.contacts.contactList.create', $data);
        } elseif ($request->method() == 'POST') {
            $validator = Validator::make($request->all(), [
                'contact_type_id' => 'required|numeric',
                'country' => 'required',
                'email' => 'required|email:rfc,dns'
            ], [
                'contact_type_id' => 'Contact Type field is required',
            ]);
            if ($validator->fails()) {
                return back()->withInput()->withErrors($validator);
            }

            $contact = new Contact();
            $fillData = $request->only($contact->getFillable());
            $fillData['user_id'] = $this->user->id;
            $custom_field = [];
            if ($data['customFields']) {
                foreach ($data['customFields'] as $item) {
                    $custom_field[$item->field_name] = [
                        'field_name' => removeHyphenInString(ucfirst($item->field_name)),
                        'field_value' => $request->{$item->field_name},
                    ];
                }
            }
            $fillData['custom_field'] = $custom_field;
            $contact->fill($fillData)->save();
            updateWallet($this->user->id, 1, 'use_contacts', 1);

            $message = 'We have finished processing your request to create contact';
            $this->emailNotification($message, route('user.contactList', $request->contact_type_id), 'View your contacts');

            $recipients = [$contact];
            $this->automationTriggered($recipients, 'contacts_', $request->contact_type_id);

            $route = route('user.contactList', $request->contact_type_id);
            $this->userActivity(
                "You have finished processing to create contact <a href='$route' class='text-primary'>$contact->email</a>!"
            );

            return back()->with('success', 'Contact Added Successfully');
        }
    }

    public function contactUpdate(Request $request, $id)
    {
        $contact = Contact::own()->with('contactType:id,name')->findOrFail($id);
        $data['customFields'] = CustomField::own()->get();
        if ($request->method() == 'GET') {
            $data['countries'] = Country::get();
            return view($this->theme . 'user.contacts.contactList.edit', $data, compact('contact'));
        } elseif ($request->method() == 'POST') {
            $validator = Validator::make($request->all(), [
                'country' => 'required',
                'email' => 'required|email'
            ]);
            if ($validator->fails()) {
                return back()->withInput()->withErrors($validator);
            }
            try {
                $custom_field = [];
                $fillData = $request->only($contact->getFillable());
                if ($data['customFields']) {
                    foreach ($data['customFields'] as $item) {
                        $custom_field[$item->field_name] = [
                            'field_name' => removeHyphenInString(ucfirst($item->field_name)),
                            'field_value' => $request->{$item->field_name},
                        ];
                    }
                }
                $fillData['custom_field'] = $custom_field;
                $contact->fill($fillData)->save();

                $this->userActivity(
                    "You have finished processing to update contact!"
                );

                return back()->with('success', 'Contact Updated Successfully');
            } catch (\Exception $e) {
                return back()->with('error', $e->getMessage());
            }
        }
    }

    public function contactDelete($id)
    {
        $contact = Contact::own()->findOrFail($id);
        $contact->delete();
        $message = 'We have finished processing your request to delete single contact';
        $this->emailNotification($message, route('user.contactList', $contact->contact_type_id), 'View your contacts');

        $route = route('user.contactList', $contact->contact_type_id);
        $this->userActivity(
            "You have finished processing to delete <a href='$route' class='text-primary'>single contact</a>!"
        );

        return back()->with('success', 'Deleted Successfully');
    }

    public function contactBulkDelete(Request $request)
    {
        if ($request->strIds == null) {
            session()->flash('error', 'You do not select ID.');
            return response()->json(['error' => 1]);
        } else {
            Contact::own()->whereIn('id', $request->strIds)->get()->map(function ($query) {
                $query->delete();
            });
            $message = 'We have finished processing your request to delete multiple contact';
            $this->emailNotification($message);

            $this->userActivity(
                "You have finished processing to delete multiple contact!"
            );

            session()->flash('success', 'Deleted Successfully');
            return response()->json(['status' => 'success']);
        }
    }

    public function contactUploadCsv(Request $request)
    {
        if ($request->method() == 'GET') {
            $data['contactTypes'] = ContactType::own(1)->get();
            return view($this->theme . 'user.contacts.contactList.uploadCSV', $data);
        } elseif ($request->method() == 'POST') {
            $validator = Validator::make($request->all(), [
                'contact_type_id' => 'required',
                'file' => 'required'
            ]);
            if ($validator->fails()) {
                return back()->withInput()->withErrors($validator);
            }

            $contactType = ContactType::own(1)->findOrFail($request->contact_type_id);
            if ($request->file->getClientOriginalExtension() != 'csv') {
                throw new \Exception('Only accepted .csv files');
            }
            $file = fopen($request->file->getRealPath(), 'r');

            $invalidEmails = [];
            $firstIteration = true;
            $count = 0;
            $contactLeft = max(auth()->user()->limit_contact - auth()->user()->use_contacts, 0);

            while ($csvLine = fgetcsv($file)) {
                if ($firstIteration) {
                    $firstIteration = false;
                    continue; // Skip the first iteration
                }

                if ($contactLeft == $count) {
                    break;
                }

                if (isset($csvLine[2])) {
                    if (!filter_var($csvLine[2], FILTER_VALIDATE_EMAIL)) {
                        $invalidEmails[] = $csvLine[2];
                        continue;
                    }

                    $contacts[] = Contact::firstOrCreate(
                        [
                            'email' => $csvLine[2],
                            'contact_type_id' => $contactType->id,
                        ],
                        [
                            'user_id' => $this->user->id,
                            'first_name' => $csvLine[0],
                            'last_name' => $csvLine[1],
                            'alt_email' => $csvLine[3],
                            'country' => $csvLine[4],
                            'state' => $csvLine[5],
                            'city' => $csvLine[6],
                            'address_line1' => $csvLine[7],
                            'address_line2' => $csvLine[8],
                            'state_province_region' => $csvLine[9],
                            'postal_code' => $csvLine[10],
                        ]
                    );
                    $count++;
                }
            }

            updateWallet($this->user->id, $count, 'use_contacts', 1);

            $message = 'We have finished processing your request to upload contacts CSV';
            $this->emailNotification($message, route('user.contactList', $contactType->id), 'View your contacts');

            $this->automationTriggered($contacts ?? [], 'contacts_', $contactType->id);

            $route = route('user.contactList', $contactType->id);
            $this->userActivity(
                "You have finished processing to upload <a href='$route' class='text-primary'>contacts CSV</a>!"
            );

            if (count($invalidEmails) > 0) {
                $msg = "Imported valid email successfully. You have " . count($invalidEmails) . " in-valid email";
                session()->put('invalidEmails', $invalidEmails);
                return back()->with('success', $msg);
            }
            session()->flash('success', 'Imported Successfully');
            return redirect()->route('user.contactList', $contactType->id);
        }
    }

    public function contactCsvSample()
    {
        $file = 'contacts-sample.csv';
        $full_path = 'assets/' . $file;
        $title = 'contacts-upload-sample';
        $ext = pathinfo($file, PATHINFO_EXTENSION);
        $mimetype = mime_content_type($full_path);
        header('Content-Disposition: attachment; filename="' . $title . '.' . $ext . '";');
        header("Content-Type: " . $mimetype);
        return readfile($full_path);
    }

    public function contactExportCsv(Request $request, $type_id)
    {
        $this->userActivity(
            "You have finished processing to export contact!"
        );
        return Excel::download(new ContactExport($request, $type_id), 'contacts.csv');
    }

    public function exportInvalidEmails(Request $request)
    {
        $this->userActivity(
            "You have finished processing to export bulk invalid-emails!"
        );
        return Excel::download(new ExportInvalidEmails($request->invalidEmails), 'invalid-emails.csv');
    }

}
