<?php

namespace App\Http\Controllers\User;

use App\Http\Controllers\Controller;
use App\Models\ApiClient;
use App\Models\Automation;
use App\Models\Deposit;
use App\Models\EmailAutomationMap;
use App\Models\EmailTemplate;
use App\Models\Gateway;
use App\Models\Kyc;
use App\Models\Language;
use App\Models\Sender;
use App\Models\SingleSend;
use App\Models\Transaction;
use App\Models\UnsubscribeGroup;
use App\Models\UserActivity;
use App\Models\UserKyc;
use Carbon\Carbon;
use Facades\App\Services\OpenAIService;
use App\Traits\Activity;
use App\Traits\Upload;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\Rule;
use Illuminate\Support\Facades\Hash;
use Illuminate\Validation\ValidationException;
use Stevebauman\Location\Facades\Location;
use Illuminate\Support\Facades\Http;
use Facades\Tintnaingwin\EmailChecker\EmailChecker;


class HomeController extends Controller
{
    use Upload, Activity;

    public function __construct()
    {
        $this->middleware(['auth']);
        $this->middleware(function ($request, $next) {
            $this->user = auth()->user();
            return $next($request);
        });
        $this->theme = template();
    }

    public function saveToken(Request $request)
    {
        Auth::user()
            ->fireBaseToken()
            ->create([
                'token' => $request->token,
            ]);
        return response()->json([
            'msg' => 'token saved successfully.',
        ]);
    }

    public function index()
    {
        $data['user'] = Auth::user();

        $thirtyDaysAgo = Carbon::now()->subDays(30);

        $data['automationRecord'] = collect(EmailAutomationMap::own()->where('status', 1)->selectRaw('COUNT(id) AS totalEmailSend')
            ->selectRaw('(COUNT(CASE WHEN triggered_date >= ? THEN id END) / COUNT(id)) * 100 AS last30DaysPercentage', [$thirtyDaysAgo])
            ->selectRaw('(COUNT(CASE WHEN unique_opens = 1 THEN id END) / COUNT(id)) * 100 AS avgOpenRate')
            ->selectRaw('(COUNT(CASE WHEN unique_opens = 1 AND open_date >= ? THEN id END) / COUNT(CASE WHEN open_date >= ? THEN id END)) * 100 AS last30DaysOpenPercentage', [$thirtyDaysAgo, $thirtyDaysAgo])
            ->selectRaw('(COUNT(CASE WHEN bounces = 1 THEN id END) / COUNT(id)) * 100 AS avgBounceRate')
            ->selectRaw('(COUNT(CASE WHEN bounces = 1 AND triggered_date >= ? THEN id END) / COUNT(CASE WHEN triggered_date >= ? THEN id END)) * 100 AS last30DaysBouncePercentage', [$thirtyDaysAgo, $thirtyDaysAgo])
            ->selectRaw('(COUNT(CASE WHEN unsubscribes = 1 THEN id END) / COUNT(id)) * 100 AS avgUnsbscribesRate')
            ->selectRaw('(COUNT(CASE WHEN unsubscribes = 1 AND triggered_date >= ? THEN id END) / COUNT(CASE WHEN triggered_date >= ? THEN id END)) * 100 AS last30DaysUnsubscribesPercentage', [$thirtyDaysAgo, $thirtyDaysAgo])
            ->get()
            ->toArray())->collapse();

        $data['singleSendRecord'] = collect(SingleSend::own()->where('status', 3)->selectRaw('SUM(triggered) AS totalEmailSend')
            ->selectRaw('(SUM(CASE WHEN sent_at >= ? THEN triggered END) / SUM(triggered)) * 100 AS last30DaysPercentage', [$thirtyDaysAgo])
            ->selectRaw('(SUM(unique_opens) / SUM(triggered)) * 100 AS avgOpenRate')
            ->selectRaw('(SUM(CASE WHEN sent_at >= ? THEN unique_opens END) / SUM(CASE WHEN sent_at >= ? THEN triggered END)) * 100 AS last30DaysOpenPercentage', [$thirtyDaysAgo, $thirtyDaysAgo])
            ->selectRaw('(SUM(bounces) / SUM(triggered)) * 100 AS avgBounceRate')
            ->selectRaw('(SUM(CASE WHEN sent_at >= ? THEN bounces END) / SUM(CASE WHEN sent_at >= ? THEN triggered END)) * 100 AS last30DaysBouncePercentage', [$thirtyDaysAgo, $thirtyDaysAgo])
            ->selectRaw('(SUM(unsubscribes) / SUM(triggered)) * 100 AS avgUnsubscribesRate')
            ->selectRaw('(SUM(CASE WHEN sent_at >= ? THEN unsubscribes END) / SUM(CASE WHEN sent_at >= ? THEN triggered END)) * 100 AS last30DaysUnsubscribesPercentage', [$thirtyDaysAgo, $thirtyDaysAgo])
            ->get()
            ->toArray())->collapse();

        $data['automation'] = collect(Automation::own()->selectRaw('COUNT(id) AS totalAutomation')
            ->selectRaw('COUNT(CASE WHEN status = 0 THEN id END) AS draftAutomation')
            ->selectRaw('COUNT(CASE WHEN status = 1 THEN id END) AS liveAutomation')
            ->selectRaw('SUM(total_triggered) AS totalTriggered')
            ->selectRaw('SUM(total_delivered) AS totalDelivered')->selectRaw('SUM(total_delivered) AS totalBounces')
            ->selectRaw('SUM(total_unique_opens) AS totalUnique_opens')->selectRaw('SUM(total_opens) AS totalTotal_opens')
            ->selectRaw('SUM(total_unsubscribe) AS totalUnsubscribes')
            ->get()
            ->toArray())->collapse();
        $data['horizontalBarChatAutomation'] = [$data['automation']['totalTriggered'], $data['automation']['totalDelivered'], $data['automation']['totalBounces'],
            $data['automation']['totalUnique_opens'], $data['automation']['totalTotal_opens'], $data['automation']['totalUnsubscribes']];

        $data['singleSend'] = collect(SingleSend::own()->selectRaw('COUNT(id) AS totalSingleSend')
            ->selectRaw('COUNT(CASE WHEN status = 0 OR status = 1 THEN id END) AS draftSingleSend')
            ->selectRaw('COUNT(CASE WHEN status = 3  THEN id END) AS completeSingleSend')
            ->selectRaw('SUM(triggered) AS totalTriggered')
            ->selectRaw('SUM(delivered) AS totalDelivered')->selectRaw('SUM(bounces) AS totalBounces')
            ->selectRaw('SUM(unique_opens) AS totalUnique_opens')->selectRaw('SUM(total_opens) AS totalTotal_opens')
            ->selectRaw('SUM(unsubscribes) AS totalUnsubscribes')
            ->get()
            ->toArray())->collapse();

        $data['horizontalBarChatSingleSend'] = [$data['singleSend']['totalTriggered'], $data['singleSend']['totalDelivered'], $data['singleSend']['totalBounces'],
            $data['singleSend']['totalUnique_opens'], $data['singleSend']['totalTotal_opens'], $data['singleSend']['totalUnsubscribes']];

        $data['senders'] = collect(Sender::own()->selectRaw('COUNT(id) AS totalSender')
            ->selectRaw('COUNT(CASE WHEN is_verified = 0 THEN id END) AS unVerifiedSender')
            ->selectRaw('COUNT(CASE WHEN is_verified = 1 THEN id END) AS verifiedSender')
            ->get()
            ->toArray())->collapse();

        $data['unsubscribeGroup'] = collect(UnsubscribeGroup::own()->selectRaw('COUNT(id) AS totalGroup')
            ->selectRaw('COUNT(CASE WHEN is_display = 0 THEN id END) AS outPreference')
            ->selectRaw('COUNT(CASE WHEN is_display = 1 THEN id END) AS inPreference')
            ->get()
            ->toArray())->collapse();

        $data['lineChartUniqueOpenSingle'] = DB::table('single_send_activities')
            ->select(DB::raw('triggered_date AS x'), DB::raw('COUNT(*) AS y'))
            ->where('type', '3')
            ->where('user_id', auth()->id())
            ->groupBy('triggered_date')
            ->get()
            ->toArray();

        $data['lineChartUniqueOpenAutomation'] = DB::table('email_automation_maps')
            ->select(DB::raw('open_date AS x'), DB::raw('COUNT(*) AS y'))
            ->where('unique_opens', 1)
            ->where('user_id', auth()->id())
            ->groupBy('open_date')
            ->get()
            ->toArray();

        $data['firebaseNotify'] = config('firebase');
        return view($this->theme . 'user.dashboard', $data);
    }

    public function kycShow($slug, $id)
    {
        $data['kycs'] = Kyc::where('status', 1)->get();
        $data['kyc'] = Kyc::where('status', 1)->findOrFail($id);
        return view($this->theme . 'user.kyc.show', $data);
    }

    public function kycVerificationSubmit(Request $request, $id)
    {
        $kyc = Kyc::where('status', 1)->findOrFail($id);
        try {
            $params = $kyc->input_form;
            $reqData = $request->except('_token', '_method');
            $rules = [];
            if ($params !== null) {
                foreach ($params as $key => $cus) {
                    $rules[$key] = [$cus->validation == 'required' ? $cus->validation : 'nullable'];
                    if ($cus->type == 'file') {
                        $rules[$key][] = 'image';
                        $rules[$key][] = 'mimes:jpeg,jpg,png';
                        $rules[$key][] = 'max:2048';
                    } elseif ($cus->type == 'text') {
                        $rules[$key][] = 'max:191';
                    } elseif ($cus->type == 'number') {
                        $rules[$key][] = 'integer';
                    } elseif ($cus->type == 'textarea') {
                        $rules[$key][] = 'min:3';
                        $rules[$key][] = 'max:300';
                    }
                }
            }

            $validator = Validator::make($reqData, $rules);
            if ($validator->fails()) {
                return back()->withErrors($validator)->withInput();
            }
            $reqField = [];
            foreach ($request->except('_token', '_method', 'type') as $k => $v) {
                foreach ($params as $inKey => $inVal) {
                    if ($k == $inKey) {
                        if ($inVal->type == 'file' && $request->hasFile($inKey)) {
                            try {
                                $file = $this->fileUpload($request[$inKey], config('filelocation.kyc.path'), null, null, 'webp', 60);
                                $reqField[$inKey] = [
                                    'field_name' => $inVal->field_name,
                                    'field_value' => $file['path'],
                                    'field_driver' => $file['driver'],
                                    'validation' => $inVal->validation,
                                    'type' => $inVal->type,
                                ];
                            } catch (\Exception $exp) {
                                session()->flash('error', 'Could not upload your ' . $inKey);
                                return back()->withInput();
                            }
                        } else {
                            $reqField[$inKey] = [
                                'field_name' => $inVal->field_name,
                                'validation' => $inVal->validation,
                                'field_value' => $v,
                                'type' => $inVal->type,
                            ];
                        }
                    }
                }
            }

            UserKyc::create([
                'user_id' => auth()->id(),
                'kyc_id' => $kyc->id,
                'kyc_type' => $kyc->name,
                'kyc_info' => $reqField
            ]);

            $route = route('user.verification.center');
            $this->userActivity(
                "You have finished processing to submit <a href='$route' class='text-primary'>$kyc->name</a> kyc!"
            );

            return back()->with('success', 'KYC Sent Successfully');
        } catch (\Exception $e) {
            return back()->with('error', $e->getMessage());
        }
    }

    public function verificationCenter()
    {
        $data['userKycs'] = UserKyc::own()->latest()->get();
        return view($this->theme . 'user.kyc.verification-center', $data);
    }

    public function apiKey(Request $request)
    {
        $user = $this->user;
        if ($request->method() == 'GET') {
            $public_key = $user->public_key;
            $secret_key = $user->secret_key;
            if (!$public_key || !$secret_key) {
                $user->public_key = bin2hex(random_bytes(20));
                $user->secret_key = bin2hex(random_bytes(20));
                $user->save();
            }
            return view($this->theme . 'user.api-key');
        } elseif ($request->method() == 'POST') {
            $user->public_key = bin2hex(random_bytes(20));
            $user->secret_key = bin2hex(random_bytes(20));
            $user->save();

            $route = route('user.apiKey');
            $this->userActivity(
                "You have finished processing to create <a href='$route' class='text-primary'>api key</a>!"
            );

            return back()->with('success', 'Api key generated successfully');
        }
    }

    public function webhookSet(Request $request)
    {
        $validator = Validator::make($request->all(), ['webhook_url' => 'required|url']);

        if ($validator->fails()) {
            return back()->withErrors($validator)->withInput();
        }
        $user = auth()->user();

        $arr = [];

        $arr[] = [
            'created_at' => Carbon::now()->format(basicControl()->date_time_format),
            'url' => $request->webhook_url,
            'code' => strRandom(5),
        ];

        if ($user->webhook_url) {
            foreach ($user->webhook_url as $url) {
                $arr[] = [
                    'created_at' => $url->created_at,
                    'url' => $url->url,
                    'code' => $url->code
                ];
            }
        }
        $user->webhook_url = $arr;
        $user->save();
        return back()->with('success', 'Webhook Url Set');
    }


    public function webhookDelete($code)
    {
        $user = auth()->user();
        $arr = [];

        if ($user->webhook_url) {
            foreach ($user->webhook_url as $url) {
                if ($url->code != $code) {
                    $arr[] = [
                        'created_at' => $url->created_at,
                        'url' => $url->url,
                        'code' => $url->code,
                    ];
                }
            }
        }
        $user->webhook_url = $arr;
        $user->save();
        return back()->with('success', 'Webhook URL Deleted');
    }


    public function addFund()
    {
        $data['basic'] = basicControl();
        $data['gateways'] = Gateway::where('status', 1)->orderBy('sort_by', 'ASC')->get();
        return view($this->theme . 'user.fund.add_fund', $data);
    }


    public function fund(Request $request)
    {
        $basic = basicControl();
        $userId = Auth::id();
        $funds = Deposit::with(['depositable', 'gateway'])
            ->where('user_id', $userId)
            ->where('payment_method_id', '>', 999)
            ->orderBy('id', 'desc')
            ->latest()->paginate($basic->paginate);
        return view($this->theme . 'user.fund.index', compact('funds'));

    }

    public function activity(Request $request)
    {
        $data['activities'] = collect();

        UserActivity::own()->with(['apiClient'])
            ->latest()
            ->chunk(1000, function ($activities) use (&$data) {
                $groupedActivities = $activities->groupBy(function ($activity) {
                    return $activity->created_at->format('F, Y');
                });

                $data['activities'] = $data['activities']->merge($groupedActivities);
            });

        return view($this->theme . 'user.activity', $data);
    }

    public function apiRecipient(Request $request)
    {
        $search = $request->all();
        $data['apiClients'] = ApiClient::own()
            ->when(isset($search['ip']), function ($query) use ($search) {
                return $query->where('ip', 'LIKE', '%' . $search['ip'] . '%');
            })
            ->get();
        return view($this->theme . 'user.apiRecipient', $data);
    }

    public function apiRecipientAction(Request $request)
    {
        $apiClient = ApiClient::own()->findOrFail($request->id);
        if ($request->action == 'blocked') {
            $apiClient->is_blocked = 1;
        } else {
            $apiClient->is_blocked = 0;
        }
        $apiClient->save();
        return back()->with('success', 'Updated Successfully');
    }

    public function openAiGenerated(Request $request)
    {
        if (!basicControl()->open_ai_status) {
            return response()->json([
                'status' => 'error',
                'generated' => 'Service is currently unavailable'
            ]);
        }
        $response = OpenAIService::textGenerator($request);
        if ($response['status'] == 'success') {
            if (count($response['data']->choices) > 0) {
                foreach ($response['data']->choices as $value) {
                    $aiRes = $value->message->content;
                }
            }
            return response()->json([
                'status' => 'success',
                'generated' => $aiRes
            ]);
        } else {
            return response()->json([
                'status' => 'error',
                'generated' => $response['message'] ?? null
            ]);
        }
    }

    public function transaction(Request $request)
    {
        $search = $request->all();
        $dateSearch = $request->datetrx;
        $date = preg_match("/^[0-9]{2,4}\-[0-9]{1,2}\-[0-9]{1,2}$/", $dateSearch);
        $userId = Auth::id();
        $transactions = Transaction::where('user_id', $userId)
            ->when(@$search['transaction_id'], function ($query) use ($search) {
                return $query->where('trx_id', 'LIKE', "%{$search['transaction_id']}%");
            })
            ->when(@$search['remark'], function ($query) use ($search) {
                return $query->where('remarks', 'LIKE', "%{$search['remark']}%");
            })
            ->when($date == 1, function ($query) use ($dateSearch) {
                return $query->whereDate("created_at", $dateSearch);
            })
            ->orderBy('id', 'desc')
            ->paginate(basicControl()->paginate);
        return view($this->theme . 'user.transaction.index', compact('transactions'));
    }

    public function emailChecker(Request $request)
    {
        $user = auth()->user();
        if ($user->credits <= 0) {
            return response()->json(['status' => 'error', 'message' => 'Your account balance is currently at zero credits. To proceed, please acquire prepaid credits through our pricing options.']);
        }
        $mailAddress = $request->checkerEmail;
        if (!filter_var($mailAddress, FILTER_VALIDATE_EMAIL)) {
            return response()->json(['status' => 'error', 'message' => 'Invalid Email Format']);
        }
        try {
            $status = EmailChecker::check($mailAddress);
            $user->credits -= 1;
            $user->save();
            return response()->json(['status' => 'success', 'message' => $status ? 'Valid' : 'Suspected Email', 'email' => $mailAddress]);

        } catch (\Exception $e) {
            return response()->json(['status' => 'error', 'message' => $e->getMessage()]);
        }
    }

    public function uploadTemplate(Request $request)
    {
        if ($request->method() == 'GET') {
            return view($this->theme . 'user.upload-template');
        } elseif ($request->method() == 'POST') {
            $validator = Validator::make($request->all(), [
                'template_name' => 'required',
                'code_type' => 'required',
                'code' => 'required',
                'preview_image' => 'required'
            ]);
            if ($validator->fails()) {
                return back()->withInput()->withErrors($validator);
            }

            try {
                $template = new EmailTemplate();
                $template->user_id = auth()->id();
                $template->template_name = $request->template_name;
                if ($request->code_type == 'html') {
                    $template->json_code = "{}";
                    $template->html_code = $request->code;
                } elseif ($request->code_type == 'json') {
                    $template->json_code = $request->code;
                }

                if ($request->file('preview_image')) {
                    $image = $this->fileUpload($request->preview_image, config('filelocation.sampleTemplate.path'), null, null, 'webp', 60, null, null);
                    if ($image) {
                        $template->preview_image = $image['path'];
                        $template->driver = $image['driver'];
                    }
                }

                $template->is_blank = 0;
                $template->custom = 1;
                $template->save();

                return back()->with('success', 'Template Uploaded Successfully');
            } catch (\Exception $e) {
                return back()->with('error', $e->getMessage());
            }
        }
    }

    public function templateHtmlSample()
    {
        $file = 'html-sample.txt';
        $full_path = 'assets/' . $file;
        $title = 'html-sample';
        $ext = pathinfo($file, PATHINFO_EXTENSION);
        $mimetype = mime_content_type($full_path);
        header('Content-Disposition: attachment; filename="' . $title . '.' . $ext . '";');
        header("Content-Type: " . $mimetype);
        return readfile($full_path);
    }

    public function templateJSONSample()
    {
        $file = 'json-sample.txt';
        $full_path = 'assets/' . $file;
        $title = 'json-sample';
        $ext = pathinfo($file, PATHINFO_EXTENSION);
        $mimetype = mime_content_type($full_path);
        header('Content-Disposition: attachment; filename="' . $title . '.' . $ext . '";');
        header("Content-Type: " . $mimetype);
        return readfile($full_path);
    }

    public function customTemplateList()
    {
        $data['templates'] = EmailTemplate::where('user_id', auth()->id())->get();
        return view($this->theme . 'user.custom-template-list', $data);
    }

    public function customTemplateDelete($id)
    {
        $template = EmailTemplate::where('user_id', auth()->id())->findOrFail($id)->delete();
        $this->fileDelete($template->driver, $template->preview_image);
        return back()->with('success', 'Deleted Successfully');
    }
}
