<?php

namespace App\Http\Controllers\User;

use App\Exports\ExportSegmentEmail;
use App\Http\Controllers\Controller;
use App\Http\Requests\SegmentEmailStoreRequest;
use App\Http\Requests\SegmentStoreRequest;
use App\Models\Contact;
use App\Models\ContactType;
use App\Models\CustomField;
use App\Models\Segment;
use App\Models\SegmentEmail;
use App\Traits\Activity;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Facades\App\Services\SegmentService;
use Illuminate\Support\Facades\DB;
use Maatwebsite\Excel\Facades\Excel;

class SegmentController extends Controller
{
    use Activity;

    public function __construct()
    {
        $this->middleware(['auth']);
        $this->middleware(function ($request, $next) {
            $this->user = auth()->user();
            return $next($request);
        });
        $this->theme = template();
    }

    public function index(Request $request)
    {
        $search = $request->all();
        $data['segments'] = Segment::own()->withCount('segmentEmails')
            ->when(isset($search['name']), function ($query) use ($search) {
                $query->where('segment_name', 'LIKE', '%' . $search['name'] . '%');
            })
            ->orderBy('id', 'desc')
            ->paginate(basicControl()->paginate);
        return view($this->theme . 'user.segment.index', $data);
    }

    public function create(Request $request)
    {
        if (isset($request->contactType)) {
            $type = ContactType::own(1)->select(['id'])->findOrFail($request->contactType);
            $data['typeId'] = $type->id;
        }
        $data['contactProfiles'] = CustomField::where('user_id', null)->get();
        $data['customFields'] = CustomField::own()->get();
        $data['contactTypes'] = ContactType::own()->get();
        return view($this->theme . 'user.segment.create', $data);
    }

    public function store(SegmentStoreRequest $request)
    {
        try {
            $queryLists = SegmentService::createSegment($request);
            if ($queryLists['status'] == 'error') {
                return back()->with('error', $queryLists['message']);
            }

            if ($queryLists['status'] == 'success') {
                $results = $queryLists['result'];
                $segment = new Segment();
                $fillData = $request->only($segment->getFillable());
                $fillData['user_id'] = $this->user->id;
                $fillData['contact_type_id'] = ($request->segment_type == 'all') ? 1 : $request->contact_type_id;
                $segment->fill($fillData)->save();

                $segmentEmails = [];
                foreach ($results as $result) {
                    $segmentEmails[] = [
                        'user_id' => $this->user->id,
                        'segment_id' => $segment->id,
                        'contact_id' => $result->id,
                        'first_name' => $result->first_name,
                        'last_name' => $result->last_name,
                        'email' => $result->email,
                        'created_at' => Carbon::now(),
                        'updated_at' => Carbon::now(),
                    ];
                }

                if (!empty($segmentEmails)) {
                    DB::table('segment_emails')->insert($segmentEmails);
                }

                $message = 'We have finished processing your request to create segment';
                $this->emailNotification($message, route('user.segmentList'), 'View your segment');

                $route = route('user.segmentList');
                $this->userActivity(
                    "You have finished processing to create <a href='$route' class='text-primary'>$segment->segment_name</a> segment!"
                );

                session()->flash('Segment Created Successfully');
                return redirect()->route('user.segmentList');
            }
            return back()->with('error', 'Something went wrong. Please try again');
        } catch (\Exception $e) {
            return back()->with('error', $e->getMessage());
        }
    }

    public function getOperator(Request $request)
    {
        $customField = CustomField::find($request->fieldValue);
        if ($customField) {
            $segmentOperators = config('segments');
            $operators = $segmentOperators[$customField->field_type];
            return response()->json(['status' => 'success', 'data' => $operators]);
        }
    }

    public function getInput(Request $request)
    {
        $customField = CustomField::find($request->fieldValue);
        if ($customField) {
            $segmentOperators = config('segments');
            $inputLists = $segmentOperators[$customField->field_type][$request->operatorValue];
            return response()->json(['status' => 'success', 'data' => $inputLists]);
        }
    }

    public function segmentDelete($id)
    {
        $segment = Segment::own()->findOrFail($id);
        $segment->delete();
        $message = 'We have finished processing your request to delete single segment';
        $this->emailNotification($message, route('user.segmentList'), 'View your segments');

        $route = route('user.segmentList');
        $this->userActivity(
            "You have finished processing to delete <a href='$route' class='text-primary'>$segment->segment_name</a> segment!"
        );

        return back()->with('success', 'Deleted Successfully');
    }

    public function segmentBulkDelete(Request $request)
    {
        if ($request->strIds == null) {
            session()->flash('error', 'You do not select ID.');
            return response()->json(['error' => 1]);
        } else {
            Segment::own()->whereIn('id', $request->strIds)->get()->map(function ($query) {
                $query->delete();
            });
            $message = 'We have finished processing your request to delete multiple segment';
            $this->emailNotification($message, route('user.segmentList'), 'View your segments');

            $route = route('user.segmentList');
            $this->userActivity(
                "You have finished processing to multiple delete <a href='$route' class='text-primary'>segment</a>!"
            );

            session()->flash('success', 'Deleted Successfully');
            return response()->json(['status' => 'success']);
        }
    }

    public function segmentEmailList(Request $request, $segmentId)
    {
        $search = $request->all();
        $data['segment'] = Segment::own()->findOrFail($segmentId);
        $data['contactEmails'] = SegmentEmail::own()->with(['contact'])->where('segment_id', $data['segment']->id)
            ->when(isset($search['firstname']), function ($query) use ($search) {
                $query->where('first_name', 'LIKE', '%' . $search['firstname'] . '%');
            })
            ->when(isset($search['lastname']), function ($query) use ($search) {
                $query->where('last_name', 'LIKE', '%' . $search['lastname'] . '%');
            })
            ->when(isset($search['email_address']), function ($query) use ($search) {
                $query->where('email', 'LIKE', '%' . $search['email_address'] . '%');
            })
            ->orderBy('id', 'desc')->paginate(basicControl()->paginate);
        return view($this->theme . 'user.segment.emailList', $data);
    }

    public function segmentEmailStore(SegmentEmailStoreRequest $request)
    {
        $segmentEmail = new SegmentEmail();
        $fillData = $request->only($segmentEmail->getFillable());
        $fillData['user_id'] = $this->user->id;
        $fillData['segment_id'] = $request->segmentId;
        $segmentEmail->fill($fillData)->save();

        $this->automationTriggered($segmentEmail ?? [], 'segments_', $segmentEmail->segment_id);

        $this->userActivity("A new segment email was created!");

        session()->flash('success', 'Added Successfully');
        return response()->json(['status' => 'success']);
    }

    public function segmentEmailDelete($id)
    {
        SegmentEmail::own()->findOrFail($id)->delete();
        $message = 'We have finished processing your request to delete segment emails';
        $this->emailNotification($message);
        $this->userActivity("You have finished processing to delete segment emails!");
        return back()->with('success', 'Deleted Successfully');
    }

    public function segmentEmailBulkDelete(Request $request)
    {
        if ($request->strIds == null) {
            session()->flash('error', 'You do not select ID.');
            return response()->json(['error' => 1]);
        } else {
            SegmentEmail::own()->whereIn('id', $request->strIds)->get()->map(function ($query) {
                $query->delete();
            });
            $message = 'We have finished processing your request to delete multiple segment emails';
            $this->emailNotification($message);
            $this->userActivity("You have finished processing to delete multiple segment emails!");
            session()->flash('success', 'Deleted Successfully');
            return response()->json(['status' => 'success']);
        }
    }

    public function segmentExportCsv(Request $request, $segmentId)
    {
        Segment::own()->findOrFail($segmentId);
        $this->userActivity("You have finished processing to export segment emails!");
        return Excel::download(new ExportSegmentEmail($request, $segmentId), 'emails.csv');
    }
}
