<?php

namespace App\Http\Controllers\User;

use App\Http\Controllers\Controller;
use App\Http\Requests\SenderSaveRequest;
use App\Mail\SenderVerifyMail;
use App\Models\City;
use App\Models\Country;
use App\Models\CountryCity;
use App\Models\CountryState;
use App\Models\Sender;
use App\Models\State;
use App\Traits\Activity;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Mail;

class SenderController extends Controller
{
    use Activity;

    public function __construct()
    {
        $this->middleware(['auth']);
        $this->middleware(function ($request, $next) {
            $this->user = auth()->user();
            return $next($request);
        });
        $this->theme = template();
    }

    public function index()
    {
        $data['senders'] = Sender::own()->latest()->get();
        $data['countries'] = Country::get();
        return view($this->theme . 'user.senders.index', $data);
    }

    public function senderSave(SenderSaveRequest $request)
    {
        DB::beginTransaction();
        try {
            $sender = new Sender();
            $fillData = $request->only($sender->getFillable());
            $fillData['user_id'] = $this->user->id;
            $fillData['utr'] = uniqid();
            $fillData['send_at'] = Carbon::now();
            $sender->fill($fillData)->save();
            Mail::to($sender->from_email)->send(new SenderVerifyMail($sender));
            DB::commit();
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json(['status' => 'error', 'message' => $e->getMessage()]);
        }

        $message = 'We have finished processing your request to add single sender';
        $this->emailNotification($message, route('user.senderList'), 'View your senders');

        $route = route('user.senderList');
        $this->userActivity(
            "You have finished processing to create <a href='$route' class='text-primary'>$sender->from_name</a> single sender!"
        );

        session()->flash('success', 'Added Successfully');
        return response()->json(['status' => 'success']);
    }

    public function senderUpdate(SenderSaveRequest $request)
    {
        $sender = Sender::own()->find($request->id);
        if ($sender) {
            $oldEmail = $sender->from_email;
            $fillData = $request->only($sender->getFillable());
            $sender->fill($fillData)->save();

            if ($oldEmail != $fillData['from_email']) {
                $sender->is_verified = 0;
                $sender->send_at = Carbon::now();
                $sender->save();
                Mail::to($fillData['from_email'])->send(new SenderVerifyMail($sender));
            }

            $route = route('user.senderList');
            $this->userActivity(
                "You have finished processing to update <a href='$route' class='text-primary'>$sender->from_name</a> single sender!"
            );

            session()->flash('success', 'Updated Successfully');
            return response()->json(['status' => 'success']);
        }
    }

    public function senderSendMail($id)
    {
        $sender = Sender::own()->findOrFail($id);
        try {
            $sender->send_at = Carbon::now();
            $sender->save();
            Mail::to($sender->from_email)->send(new SenderVerifyMail($sender));

            $route = route('user.senderList');
            $this->userActivity(
                "You have finished processing to send <a href='$route' class='text-primary'>$sender->from_name</a> Verification Mail!"
            );

            return back()->with('success', 'Verification Mail Send');
        } catch (\Exception $e) {
            return back()->with('error', $e->getMessage());
        }
    }

    public function senderDelete($id)
    {
        $sender = Sender::own()->findOrFail($id);
        $sender->delete();
        $message = 'We have finished processing your request to delete single sender';
        $this->emailNotification($message, route('user.senderList'), 'View your senders');

        $route = route('user.senderList');
        $this->userActivity(
            "You have finished processing to delete <a href='$route' class='text-primary'>$sender->from_name</a> single sender!"
        );

        return back()->with('success', 'Deleted Successfully');
    }

    public function getState(Request $request)
    {
        $data['states'] = CountryState::where("country_id", $request->country_id)->get(["name", "id"]);
        return response()->json($data);
    }

    public function getCity(Request $request)
    {
        $data['cities'] = CountryCity::where("state_id", $request->state_id)->get(["name", "id"]);
        return response()->json($data);
    }
}
