<?php

namespace App\Http\Controllers\User;

use App\Exports\ExportRecipient;
use App\Exports\GlobalUnsubscribesExport;
use App\Http\Controllers\Controller;
use App\Http\Requests\UnsubscribesGroupRequest;
use App\Http\Requests\UnsubscribesStore;
use App\Models\EmailTemplate;
use App\Models\UnsubscribeClient;
use App\Models\UnsubscribeGroup;
use App\Traits\Activity;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Maatwebsite\Excel\Facades\Excel;

class UnsubscribeController extends Controller
{
    use Activity;

    public function __construct()
    {
        $this->middleware(['auth']);
        $this->middleware(function ($request, $next) {
            $this->user = auth()->user();
            return $next($request);
        });
        $this->theme = template();
    }

    public function globalUnsubscribes(Request $request)
    {
        $search = $request->all();
        $data['unSubscribers'] = UnsubscribeClient::own()->orderBy('id', 'desc')
            ->when(isset($search['daterange']), function ($query) use ($search) {
                list($startDate, $endDate) = explode(' - ', $search['daterange']);
                $startDate = Carbon::parse($startDate)->startOfDay();
                $endDate = Carbon::parse($endDate)->endOfDay();
                $query->whereBetween('created_at', [$startDate, $endDate]);
            })
            ->when(isset($search['email_address']), function ($query) use ($search) {
                $query->where('email_address', 'LIKE', '%' . $search['email_address'] . '%');
            })
            ->paginate(basicControl()->paginate);
        return view($this->theme . 'user.unsubscribes.globalIndex', $data);
    }

    public function globalUnsubscribesSave(UnsubscribesStore $request)
    {
        $emailString = $request->email_address;
        $emailArray = explode(',', $emailString);
        foreach ($emailArray as $email) {
            $unsubscribeClients = new UnsubscribeClient();
            $fillData = $request->only($unsubscribeClients->getFillable());
            $fillData['user_id'] = $this->user->id;
            $fillData['email_address'] = $email;
            $unsubscribeClients->fill($fillData)->save();
        }

        $this->userActivity(
            "A new global unsubscribers was created!"
        );

        session()->flash('success', 'Added Successfully');
        return response()->json(['status' => 'success']);
    }

    public function globalUnsubscribesDelete($id)
    {
        $unsubscribes = UnsubscribeClient::own()->findOrFail($id);
        $unsubscribes->delete();
        $message = 'We have finished processing your request to delete unsubscribers';
        $this->emailNotification($message, route('user.globalUnsubscribes'), 'View your unsubscribers');

        $this->userActivity(
            "A single global unsubscribers was deleted!"
        );

        return back()->with('success', 'Deleted Successfully');
    }

    public function globalUnsubscribesBulkDelete(Request $request)
    {
        if ($request->strIds == null) {
            session()->flash('error', 'You do not select ID.');
            return response()->json(['error' => 1]);
        } else {
            UnsubscribeClient::own()->whereIn('id', $request->strIds)->get()->map(function ($query) {
                $query->delete();
            });
            $message = 'We have finished processing your request to delete multiple unsubscribers';
            $this->emailNotification($message, route('user.globalUnsubscribes'), 'View your unsubscribers');

            $this->userActivity(
                "Multiple  global unsubscribers was deleted!"
            );

            session()->flash('success', 'Deleted Successfully');
            return response()->json(['status' => 'success']);
        }
    }

    public function globalUnsubscribesExport(Request $request)
    {
        $this->userActivity(
            "Global unsubscribers was exported!"
        );
        return Excel::download(new GlobalUnsubscribesExport($request), 'unsubscribers.csv');
    }

    public function unsubscribesGroup()
    {
        $data['unsubscribesGroups'] = UnsubscribeGroup::own()->withCount('UnsubscribeClients')->orderBy('id', 'desc')->get();
        return view($this->theme . 'user.unsubscribes.group', $data);
    }

    public function unsubscribesGroupCreate()
    {
        $data['unsubscribesGroups'] = UnsubscribeGroup::own()->where('is_display', 1)->orderBy('id', 'desc')->get();
        return view($this->theme . 'user.unsubscribes.groupCreate', $data);
    }

    public function unsubscribesGroupSave(UnsubscribesGroupRequest $request)
    {
        $unsubscribeGroup = new UnsubscribeGroup();
        $fillData = $request->only($unsubscribeGroup->getFillable());
        $fillData['user_id'] = $this->user->id;
        $fillData['rand_id'] = rand(00000, 99999);
        $unsubscribeGroup->fill($fillData)->save();

        $message = 'We have finished processing your request to add unsubscribe group';
        $this->emailNotification($message, route('user.unsubscribesGroup'), 'View your groups');

        $route = route('user.unsubscribesGroup');
        $this->userActivity(
            "you have finished processing to create <a href='$route' class='text-primary'>$unsubscribeGroup->group_name</a> unsubscribe group!"
        );

        session()->flash('success', 'Saved Successfully');
        return response()->json(['status' => 'success', 'route' => route('user.unsubscribesGroup')]);
    }

    public function unsubscribesGroupEdit($id)
    {
        $data['unsubscribesGroup'] = UnsubscribeGroup::own()->findOrFail($id);
        $data['unsubscribesGroups'] = UnsubscribeGroup::own()->where('is_display', 1)->where('id', '!=', $id)->orderBy('id', 'desc')->get();
        return view($this->theme . 'user.unsubscribes.groupEdit', $data);
    }

    public function unsubscribesGroupUpdate(UnsubscribesGroupRequest $request, $id)
    {
        $unsubscribeGroup = UnsubscribeGroup::own()->find($id);
        if ($unsubscribeGroup) {
            $fillData = $request->only($unsubscribeGroup->getFillable());
            $unsubscribeGroup->fill($fillData)->save();

            $message = 'We have finished processing your request to update unsubscribe group';
            $this->emailNotification($message, route('user.unsubscribesGroup'), 'View your groups');

            $route = route('user.unsubscribesGroup');
            $this->userActivity(
                "you have finished processing to update <a href='$route' class='text-primary'>$unsubscribeGroup->group_name</a> unsubscribe group!"
            );

            session()->flash('success', 'Updated Successfully');
            return response()->json(['status' => 'success', 'route' => route('user.unsubscribesGroup')]);
        }
    }

    public function unsubscribesGroupManuallyAdd(UnsubscribesStore $request)
    {
        $group = UnsubscribeGroup::own()->find($request->unsubscribe_groups_id);
        if ($group) {
            $emailString = $request->email_address;
            $emailArray = explode(',', $emailString);
            foreach ($emailArray as $email) {
                $unsubscribeClients = new UnsubscribeClient();
                $fillData = $request->only($unsubscribeClients->getFillable());
                $fillData['user_id'] = $this->user->id;
                $fillData['unsubscribe_groups_id'] = $group->id;
                $fillData['email_address'] = $email;
                $unsubscribeClients->fill($fillData)->save();
            }

            $this->userActivity(
                "Unsubscribe group recipients was added!"
            );

            session()->flash('success', 'Added Successfully');
            return response()->json(['status' => 'success']);
        }
    }

    public function unsubscribesGroupUploadCsv(Request $request, $id)
    {
        $unsubscribesGroup = UnsubscribeGroup::own()->select(['id', 'group_name'])->findOrFail($id);
        if ($request->method() == 'GET') {
            return view($this->theme . 'user.unsubscribes.uploadCSV', compact('unsubscribesGroup'));
        } elseif ($request->method() == 'POST') {
            if ($request->file->getClientOriginalExtension() != 'csv') {
                throw new \Exception('Only accepted .csv files');
            }
            $file = fopen($request->file->getRealPath(), 'r');

            $firstIteration = true;
            while ($csvLine = fgetcsv($file)) {
                if ($firstIteration) {
                    $firstIteration = false;
                    continue; // Skip the first iteration
                }
                UnsubscribeClient::firstOrCreate(
                    [
                        'email_address' => $csvLine[0],
                        'unsubscribe_groups_id' => $unsubscribesGroup->id,
                    ],
                    [
                        'user_id' => $this->user->id,
                    ]
                );
            }
            $message = 'We have finished processing your request to upload unsubscribes recipients CSV';
            $this->emailNotification($message, route('user.unsubscribesRecipients', $unsubscribesGroup->id), 'View your recipients');

            $route = route('user.unsubscribesRecipients', $unsubscribesGroup->id);
            $this->userActivity(
                "you have finished processing to upload <a href='$route' class='text-primary'>unsubscribes recipients CSV</a>!"
            );

            session()->flash('success', 'Imported Successfully');
            return redirect()->route('user.unsubscribesRecipients', $unsubscribesGroup->id);
        }
    }

    public function unsubscribesGroupsCsvSample()
    {
        $file = 'unsubscribe-sample.csv';
        $full_path = 'assets/' . $file;
        $title = 'unsubscribe-upload-sample';
        $ext = pathinfo($file, PATHINFO_EXTENSION);
        $mimetype = mime_content_type($full_path);
        header('Content-Disposition: attachment; filename="' . $title . '.' . $ext . '";');
        header("Content-Type: " . $mimetype);
        return readfile($full_path);
    }

    public function unsubscribesGroupsExportCsv($id)
    {
        $this->userActivity(
            "Unsubscribe group recipients was exported!"
        );
        return Excel::download(new ExportRecipient($id), 'recipients.csv');
    }

    public function unsubscribesRecipients(Request $request, $groupId)
    {
        $search = $request->all();
        $data['group'] = UnsubscribeGroup::own()->findOrFail($groupId);
        $data['recipients'] = UnsubscribeClient::own()->where('unsubscribe_groups_id', $groupId)
            ->when(isset($search['email']), function ($query) use ($search) {
                $query->where('email_address', 'LIKE', '%' . $search['email'] . '%');
            })
            ->orderBy('id', 'desc')
            ->paginate(basicControl()->paginate);
        return view($this->theme . 'user.unsubscribes.recipients', $data);
    }

    public function unsubscribesRecipientDelete($id)
    {
        $recipient = UnsubscribeClient::own()->findOrFail($id);
        $recipient->delete();
        $message = 'We have finished processing your request to delete unsubscribes recipients';
        $this->emailNotification($message, route('user.unsubscribesRecipients', $recipient->unsubscribe_groups_id), 'View your recipients');

        $route = route('user.unsubscribesRecipients', $recipient->unsubscribe_groups_id);
        $this->userActivity(
            "you have finished processing to delete <a href='$route' class='text-primary'>unsubscribes recipients</a>!"
        );

        return back()->with('success', 'Deleted Successfully');
    }

    public function unsubscribesRecipientBulkDelete(Request $request)
    {
        if ($request->strIds == null) {
            session()->flash('error', 'You do not select ID.');
            return response()->json(['error' => 1]);
        } else {
            UnsubscribeClient::own()->whereIn('id', $request->strIds)->get()->map(function ($query) {
                $query->delete();
            });
            $message = 'We have finished processing your request to delete multiple unsubscribes recipients';
            $this->emailNotification($message);

            $this->userActivity("you have finished processing to unsubscribe group recipients multiple delete!");

            session()->flash('success', 'Deleted Successfully');
            return response()->json(['status' => 'success']);
        }
    }

}
