<?php

namespace App\Models;

use App\Traits\TimeZone;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class SingleSend extends Model
{
    use HasFactory, TimeZone;

    protected $casts = [
        'recipient_id' => 'object',
        'exclude_specific_recipient_id' => 'object',
        'unique_open_token' => 'array',
    ];

    public function scopeOwn($query)
    {
        return $query->where('user_id', auth()->id());
    }

    public function user()
    {
        return $this->belongsTo(User::class, 'user_id');
    }

    public function sender()
    {
        return $this->belongsTo(Sender::class, 'sender_id')->withTrashed();
    }

    public function unsubscribeGroup()
    {
        return $this->belongsTo(UnsubscribeGroup::class, 'unsubscribe_group_id');
    }

    public function template()
    {
        return $this->belongsTo(EmailTemplate::class, 'template_id');
    }

    public function singleSendActivities()
    {
        return $this->hasMany(SingleSendActivity::class, 'single_send_id');
    }

    public function singleSendActivitiesUnsubscribes()
    {
        return $this->hasMany(SingleSendActivity::class, 'single_send_id')->where('type', 4);
    }

    public function getPercent()
    {
        $this->unsubscribes = count($this->singleSendActivitiesUnsubscribes);
        $this->save();
        return [
            'deliveredPercent' => ($this->triggered ? getAmount((100 * $this->delivered) / $this->triggered, 2) : 0),
            'bouncesPercent' => ($this->triggered ? getAmount((100 * $this->bounces) / $this->triggered, 2) : 0),
            'uniqueOpenPercent' => ($this->delivered ? getAmount((100 * $this->unique_opens) / $this->delivered, 2) : 0),
            'unsubscribesPercent' => ($this->delivered ? getAmount((100 * $this->unsubscribes) / $this->delivered, 2) : 0),
        ];
    }

    public function recipientEmails()
    {
        $filteredItems = [];
        if (count($this->recipient_id) > 0) {
            foreach ($this->recipient_id as $recipient) {
                if ($recipient->table == 'contacts_') {
                    $contacts = [];
                    $recipients[] = Contact::where('contact_type_id', $recipient->id)->where('user_id', $this->user_id)
                        ->get()->map(function ($query) use ($contacts) {
                        $contacts['first_name'] = $query->first_name;
                        $contacts['last_name'] = $query->last_name;
                        $contacts['email'] = $query->email;
                        $contacts['alt_email'] = $query->alt_email;
                        $contacts['country'] = $query->country;
                        $contacts['state'] = $query->state;
                        $contacts['city'] = $query->city;
                        $contacts['address_line1'] = $query->address_line1;
                        $contacts['address_line2'] = $query->address_line2;
                        $contacts['state_province_region'] = $query->state_province_region;
                        $contacts['postal_code'] = $query->postal_code;

                        if ($query->custom_field) {
                            foreach ($query->custom_field as $key => $field) {
                                $contacts[$key] = $field->field_value;
                            }
                        }

                        return $contacts;
                    });
                } elseif ($recipient->table == 'segments_') {
                    $segments = [];
                    $recipients[] = SegmentEmail::with(['contact'])->where('segment_id', $recipient->id)
                        ->where('user_id', $this->user_id)->get()->map(function ($query) use ($segments) {
                        $segments['first_name'] = $query->first_name;
                        $segments['last_name'] = $query->last_name;
                        $segments['email'] = $query->email;
                        $segments['alt_email'] = optional($query->contact)->alt_email;
                        $segments['country'] = optional($query->contact)->country;
                        $segments['state'] = optional($query->contact)->state;
                        $segments['city'] = optional($query->contact)->city;
                        $segments['address_line1'] = optional($query->contact)->address_line1;
                        $segments['address_line2'] = optional($query->contact)->address_line2;
                        $segments['state_province_region'] = optional($query->contact)->state_province_region;
                        $segments['postal_code'] = optional($query->contact)->postal_code;

                        if (isset($query->contact->custom_field)) {
                            foreach ($query->contact->custom_field as $key => $field) {
                                $segments[$key] = $field->field_value;
                            }
                        }

                        return $segments;
                    });
                }
            }
            $itemsList = collect($recipients)->collapse()->unique('email');
            $excludedEmails = $this->excludeRecipientEmails();
            $unsubscribesEmail = $this->unsubscribesEmail();

            $filteredItems = $itemsList->whereNotIn('email', $excludedEmails)->whereNotIn('email', $unsubscribesEmail);
        }
        return $filteredItems;
    }

    public function excludeRecipientEmails()
    {
        $recipients = [];
        if (count($this->exclude_specific_recipient_id) > 0) {
            foreach ($this->exclude_specific_recipient_id as $recipient) {
                if ($recipient->table == 'contacts_') {
                    $recipients[] = Contact::where('contact_type_id', $recipient->id)->pluck('email');
                } elseif ($recipient->table == 'segments_') {
                    $recipients[] = SegmentEmail::with(['contact'])->where('segment_id', $recipient->id)->pluck('email');
                }
            }
            $recipients = collect($recipients)->collapse()->unique();
        }
        return $recipients;
    }

    public function unsubscribesEmail()
    {
        $unId = ($this->unsubscribe_group_id == '0') ? null : $this->unsubscribe_group_id;
        $unsubscribeEmails = UnsubscribeClient::where('unsubscribe_groups_id', $unId)->pluck('email_address')->unique();
        return $unsubscribeEmails;
    }

    public function getSentAtAttribute($value)
    {
        return $this->localDateFormat($value);
    }

}
